% ************************************************************************
%   Description:
%   Implementation of Capitaine et al. (2002) series for X, Y and s+XY/2,
%   consistent with the IAU 2000A precession-nutation model. Based on SOFA
%   routine XYS2000A.f.
% 
%   Reference: 
%   IERS Conventions 2000, Chapter 5
%
%   Input:										
%      MJD      (n,1)      TT date mod. jul. date [d]
%                
%   Output:
%      X        (n,1)      X-coordinate of CIP unit vector in GCRS [rad]
%      Y        (n,1)      Y-coordinate of CIP unit vector in GCRS [rad]
%      S        (n,1)      the quantity s, positioning the CEO [rad]
% 
%   External calls: 	
%      fund_arg.m, as2rad.m
%
%   Coded for VieVS: 
%   10 Jan 2009 by Lucia Plank
%
%   Revision: 
% 
% *************************************************************************
function [X,Y,S] = xys2000a (MJD)

% ------------------------------
%  prepare Time argument
% ------------------------------
                          
        T   = (MJD-51544.5)./36525;  % time since J2000 in jul .centuries
        
        T2 = T.^2;
        T3 = T.^3;
        T4 = T.^4;
        T5 = T.^5;
        
        n  = length (T);
        
% --------------------------------
%  initialize variables / vectors
% --------------------------------

      E = zeros(1,n)+1;

% ----------------------------------
%  The series for X: numerical values
% ----------------------------------
 
%   Polynomial coefficients [as]
      Xpol = [     -0.01661699, ...
                 2004.19174288, ...
                   -0.42721905, ...
                   -0.19862054, ...
                   -0.00004605, ...
                    0.00000598 ];

% Expression for the X coordinate of the CIP in the GCRS based on the IAU2000A 
% precession-nutation model 

% --------------------------------------------------------------------------------------------------------------
% 
% X = polynomial part + non-polynomial part
% 
% Non-polynomial part (unit microarcsecond)
% (ARG being for various combination of the fundamental arguments of the nutation theory)
% 
%   Sum_i[a_{s,0})_i * sin(ARG) + a_{c,0})_i * cos(ARG)] 
% 
% + Sum_i)j=1,4 [a_{s,j})_i * t^j * sin(ARG) + a_{c,j})_i * cos(ARG)] * t^j]
% 
% The Table below provides the values for a_{s,j})_i and a_{c,j})_i
% 
% The expressions for the fundamental arguments appearing in columns 4 to 8 (luni-solar part) 
% and in columns 6 to 17 (planetary part) are those of the IERS Conventions 2000
% 
% --------------------------------------------------------------------------------------------------------------
% 
%     i    a_{s,j})_i      a_{c,j})_i    l    l'   F    D   Om L_Me L_Ve  L_E L_Ma  L_J L_Sa  L_U L_Ne  p_A
% 
% --------------------------------------------------------------------------------------------------------------
% 
% j = 0  Nb of terms = 1306        
a0=[...
    1    -6844318.44        1328.67    0    0    0    0    1    0    0    0    0    0    0    0    0    0
    2     -523908.04        -544.76    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
    3      -90552.22         111.23    0    0    2    0    2    0    0    0    0    0    0    0    0    0
    4       82168.76         -27.64    0    0    0    0    2    0    0    0    0    0    0    0    0    0
    5       58707.02         470.05    0    1    0    0    0    0    0    0    0    0    0    0    0    0
    6       28288.28         -34.69    1    0    0    0    0    0    0    0    0    0    0    0    0    0
    7      -20557.78         -20.84    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
    8      -15406.85          15.12    0    0    2    0    1    0    0    0    0    0    0    0    0    0
    9      -11991.74          32.46    1    0    2    0    2    0    0    0    0    0    0    0    0    0
   10       -8584.95           4.42    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
   11       -6245.02          -6.68    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
   12        5095.50           7.19    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
   13       -4910.93           0.76    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
   14        2521.07          -5.97    0    0    0    2    0    0    0    0    0    0    0    0    0    0
   15        2511.85           1.07    1    0    0    0    1    0    0    0    0    0    0    0    0    0
   16        2372.58           5.93    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
   17        2307.58          -7.52    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
   18       -2053.16           5.13    1    0    2    0    1    0    0    0    0    0    0    0    0    0
   19        1898.27          -0.72    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
   20       -1825.49           1.23    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   21       -1534.09           6.29    0    0    2    2    2    0    0    0    0    0    0    0    0    0
   22       -1292.02           0.00    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
   23       -1234.96           5.21    2    0    2    0    2    0    0    0    0    0    0    0    0    0
   24        1163.22          -2.94    2    0    0    0    0    0    0    0    0    0    0    0    0    0
   25        1137.48          -0.04    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
   26        1029.70          -2.63    0    0    2    0    0    0    0    0    0    0    0    0    0    0
   27        -866.48           0.52    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
   28        -813.13           0.40    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   29         664.57          -0.40    0    2    0    0    0    0    0    0    0    0    0    0    0    0
   30        -628.24          -0.64    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
   31        -603.52           0.44    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
   32        -556.26           3.16    0    1    0    0    1    0    0    0    0    0    0    0    0    0
   33        -512.37          -1.47    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
   34         506.65           2.54    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
   35         438.51          -0.56    2    0   -2    0    0    0    0    0    0    0    0    0    0    0
   36         405.91           0.99    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
   37        -122.67         203.78    0    0    1   -1    1    0    0   -1    0   -2    5    0    0    0
   38        -305.78           1.75    1    0    2    2    2    0    0    0    0    0    0    0    0    0
   39         300.99          -0.44    0    1    2    0    2    0    0    0    0    0    0    0    0    0
   40        -292.37          -0.32    1    1    0   -2    0    0    0    0    0    0    0    0    0    0
   41         284.09           0.32    0    1   -2    0   -2    0    0    0    0    0    0    0    0    0
   42        -264.02           0.99    0    0    2    2    1    0    0    0    0    0    0    0    0    0
   43         261.54          -0.95    1    0    0    2    0    0    0    0    0    0    0    0    0    0
   44         256.30          -0.28    2    0    2   -2    2    0    0    0    0    0    0    0    0    0
   45        -250.54           0.08    0    0    0    2    1    0    0    0    0    0    0    0    0    0
   46         230.72           0.08    1    0    2   -2    1    0    0    0    0    0    0    0    0    0
   47         229.78          -0.60    2    0    0   -2   -1    0    0    0    0    0    0    0    0    0
   48        -212.82           0.84    2    0    2    0    1    0    0    0    0    0    0    0    0    0
   49         196.64          -0.84    0    0    0    2   -1    0    0    0    0    0    0    0    0    0
   50         188.95          -0.12    0    1   -2    2   -1    0    0    0    0    0    0    0    0    0
   51         187.95          -0.24    1   -1    0    0    0    0    0    0    0    0    0    0    0    0
   52        -160.15         -14.04    1    0    0   -1    0    0    0    0    0    0    0    0    0    0
   53        -172.95          -0.40    0    1    0   -2    0    0    0    0    0    0    0    0    0    0
   54        -168.26           0.20    0    0    0    1    0    0    0    0    0    0    0    0    0    0
   55         161.79           0.24    2    0    0   -2    1    0    0    0    0    0    0    0    0    0
   56         161.34           0.20    1    0   -2    0    0    0    0    0    0    0    0    0    0    0
   57          57.44          95.82    0    0    0    0    0    0    0    0    0    2   -5    0    0   -1
   58         142.16           0.20    0    1    2   -2    1    0    0    0    0    0    0    0    0    0
   59        -134.81           0.20    1    1    0    0    0    0    0    0    0    0    0    0    0    0
   60         132.81          -0.52    1    0    2    0    0    0    0    0    0    0    0    0    0    0
   61        -130.31           0.04    1   -1    0   -1    0    0    0    0    0    0    0    0    0    0
   62         121.98          -0.08    2    0   -2    0   -2    0    0    0    0    0    0    0    0    0
   63        -115.40           0.60    3    0    2    0    2    0    0    0    0    0    0    0    0    0
   64        -114.49           0.32    1   -1    2    0    2    0    0    0    0    0    0    0    0    0
   65         112.14           0.28    1    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
   66         105.29           0.44    0    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
   67          98.69          -0.28    1    1    2    0    2    0    0    0    0    0    0    0    0    0
   68          91.31          -0.40    2    0    0    0   -1    0    0    0    0    0    0    0    0    0
   69          86.74          -0.08    2    0    0    0    1    0    0    0    0    0    0    0    0    0
   70         -18.38          63.80    0    0    0    0    0    0    0    4   -8    3    0    0    0    0
   71          82.14           0.00    0    0    0    0    0    0    3   -5    0    0    0    0    0   -2
   72          79.03          -0.24    1    0   -2    2   -1    0    0    0    0    0    0    0    0    0
   73           0.00         -79.08    0    1   -1    1   -1    0    0    0    0    0    0    0    0    0
   74         -78.56           0.00    1    0    0    0    2    0    0    0    0    0    0    0    0    0
   75          47.73          23.79    0    0    1   -1    1    0   -8   12    0    0    0    0    0    0
   76          66.03          -0.20    0    0    2    1    2    0    0    0    0    0    0    0    0    0
   77          62.65          -0.24    3    0    0    0    0    0    0    0    0    0    0    0    0    0
   78          60.50           0.36    1    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
   79          59.07           0.00    0    0    0    0    0    0    1   -1    0    0    0    0    0    0
   80          57.28           0.00    0    0    0    0    0    0    0    8  -16    4    5    0    0    0
   81         -55.66           0.16    1    0    0    0   -2    0    0    0    0    0    0    0    0    0
   82         -54.81          -0.08    2    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
   83         -53.22          -0.20    1    0    0   -4    0    0    0    0    0    0    0    0    0    0
   84         -52.95           0.32    1    0    2    2    1    0    0    0    0    0    0    0    0    0
   85         -52.27           0.00    1   -1    0   -1   -1    0    0    0    0    0    0    0    0    0
   86          51.32           0.00    1    1    2   -2    2    0    0    0    0    0    0    0    0    0
   87         -51.00          -0.12    2    0    0   -4    0    0    0    0    0    0    0    0    0    0
   88          51.02           0.00    0    2   -2    2   -1    0    0    0    0    0    0    0    0    0
   89         -48.65          -1.15    0    0    0    0    0    0    0    1    0   -1    0    0    0    0
   90          48.29           0.20    2    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
   91         -46.38           0.00    0    0    0    0    0    0    0    0    0    2    0    0    0    2
   92         -45.59          -0.12    1    0   -4    0   -2    0    0    0    0    0    0    0    0    0
   93         -43.76           0.36    2    0    2    2    2    0    0    0    0    0    0    0    0    0
   94         -40.58          -1.00    1    0    0   -1   -1    0    0    0    0    0    0    0    0    0
   95           0.00         -41.53    1    0   -1    0   -1    0    0    0    0    0    0    0    0    0
   96          40.54          -0.04    2    0    2   -2    1    0    0    0    0    0    0    0    0    0
   97          40.33          -0.04    2    1    0   -2    0    0    0    0    0    0    0    0    0    0
   98         -38.57           0.08    1    0    0    2    1    0    0    0    0    0    0    0    0    0
   99          37.75           0.04    1   -1    0   -2    0    0    0    0    0    0    0    0    0    0
  100          37.15          -0.12    3    0    2   -2    2    0    0    0    0    0    0    0    0    0
  101          36.68          -0.04    0    0    4   -2    2    0    0    0    0    0    0    0    0    0
  102         -18.30         -17.30    0    0    0    0    1    0    0   -1    2    0    0    0    0    0
  103         -17.86          17.10    0    0    0    0    0    0    0    1   -2    0    0    0    0    0
  104         -34.81           0.04    0    1   -2    2    0    0    0    0    0    0    0    0    0    0
  105         -33.22           0.08    0    0    2   -2   -1    0    0    0    0    0    0    0    0    0
  106          32.43          -0.04    0    1    2    0    1    0    0    0    0    0    0    0    0    0
  107         -30.47           0.04    1    0    2   -2    0    0    0    0    0    0    0    0    0    0
  108         -29.53           0.04    1    1    0   -2   -1    0    0    0    0    0    0    0    0    0
  109          28.50          -0.08    2    0   -2    0    1    0    0    0    0    0    0    0    0    0
  110          28.35          -0.16    0    1    0    0    2    0    0    0    0    0    0    0    0    0
  111         -28.00           0.00    0    0    2   -1    2    0    0    0    0    0    0    0    0    0
  112         -27.61           0.20    0    0    2    4    2    0    0    0    0    0    0    0    0    0
  113         -26.77           0.08    0    1    0    2    0    0    0    0    0    0    0    0    0    0
  114          26.54          -0.12    0    0    2    0   -1    0    0    0    0    0    0    0    0    0
  115          26.54           0.04    0    1   -2    0   -1    0    0    0    0    0    0    0    0    0
  116         -26.17           0.00    0    1    2   -2    0    0    0    0    0    0    0    0    0    0
  117         -25.42          -0.08    1    0   -2   -2    0    0    0    0    0    0    0    0    0    0
  118         -16.91           8.43    0    0    0    0    0    0    8  -13    0    0    0    0    0   -1
  119           0.32          24.42    0    0    0    0    0    0    2   -3    0    0    0    0    0    0
  120         -19.53           5.09    0    0    0    0    0    0    0    0    0    2   -5    0    0    0
  121         -23.79           0.00    0    0    0    0    0    0    2   -2    0    0    0    0    0    0
  122          23.66           0.00    1   -1    0   -1   -2    0    0    0    0    0    0    0    0    0
  123         -23.47           0.16    1   -1    2    2    2    0    0    0    0    0    0    0    0    0
  124          23.39          -0.12    2    0    0    2    0    0    0    0    0    0    0    0    0    0
  125         -23.49           0.00    0    1    0    0   -2    0    0    0    0    0    0    0    0    0
  126         -23.28          -0.08    1    1    0   -2    1    0    0    0    0    0    0    0    0    0
  127         -22.99           0.04    1    0   -2    2    0    0    0    0    0    0    0    0    0    0
  128         -22.67          -0.08    1   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  129           9.35          13.29    0    0    0    0    0    0    8  -13    0    0    0    0    0    0
  130          22.47          -0.04    0    1    0    1    0    0    0    0    0    0    0    0    0    0
  131           4.89         -16.55    0    0    0    0    0    0    0    2   -8    3    0    0    0   -2
  132           4.89         -16.51    0    0    0    0    0    0    0    6   -8    3    0    0    0    2
  133          21.28          -0.08    0    1    2    2    2    0    0    0    0    0    0    0    0    0
  134          20.57           0.64    0    0    0    0    0    0    0    3    0   -1    0    0    0    2
  135          21.01           0.00    1   -1    0    0    1    0    0    0    0    0    0    0    0    0
  136           1.23         -19.13    0    0    1   -1    1    0    0   -1    0    2   -5    0    0    0
  137         -19.97           0.12    3    0    2    0    1    0    0    0    0    0    0    0    0    0
  138          19.65          -0.08    0    0    0    4    0    0    0    0    0    0    0    0    0    0
  139          19.58          -0.12    1    0    0    2   -1    0    0    0    0    0    0    0    0    0
  140          19.61          -0.08    1   -1    0    2    0    0    0    0    0    0    0    0    0    0
  141         -19.41           0.08    2   -1    2    0    2    0    0    0    0    0    0    0    0    0
  142         -19.49           0.00    0    0    0    0    0    0    4   -6    0    0    0    0    0   -2
  143         -18.64           0.00    0    0    0    2    2    0    0    0    0    0    0    0    0    0
  144          18.58           0.04    1    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  145         -18.42           0.00    1   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
  146          18.22           0.00    0    0    0    0    0    0    2   -4    0    0    0    0    0   -2
  147          -0.72         -17.34    0    0    2   -2    1    0   -5    6    0    0    0    0    0    0
  148         -18.02          -0.04    1    0    2   -4    1    0    0    0    0    0    0    0    0    0
  149          17.74           0.08    0    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  150          17.46           0.00    2    0    0   -2    0    0    0   -2    0    2    0    0    0    0
  151         -17.42           0.00    0    3    2   -2    2    0    0    0    0    0    0    0    0    0
  152          -6.60          10.70    0    0    0    0    0    0    0    1    0   -2    0    0    0    0
  153          16.43           0.52    0    0    0    0    0    0    0    2    0   -2    0    0    0    0
  154         -16.75           0.04    1   -1    2    0    1    0    0    0    0    0    0    0    0    0
  155          16.55          -0.08    0    0    2    2    0    0    0    0    0    0    0    0    0    0
  156          16.39          -0.08    2    1    2    0    2    0    0    0    0    0    0    0    0    0
  157          13.88          -2.47    2    0    0   -2    0    0    0   -2    0    3    0    0    0    0
  158          15.69           0.00    1    0    0   -2   -2    0    0    0    0    0    0    0    0    0
  159         -15.52           0.00    0    0    0    1    1    0    0    0    0    0    0    0    0    0
  160           3.34          11.86    0    0    0    0    1    0    0   -4    8   -3    0    0    0    0
  161          14.72          -0.32    0    0    0    0    0    0    2    0    0    0    0    0    0    2
  162          14.92          -0.04    2   -1    0    0    0    0    0    0    0    0    0    0    0    0
  163          -3.26          11.62    0    0    0    0    1    0    0    4   -8    3    0    0    0    0
  164         -14.64           0.00    0    0    0    0    0    0    0    2   -2    0    0    0    0    0
  165           0.00          14.47    1    0   -1    0   -2    0    0    0    0    0    0    0    0    0
  166         -14.37           0.00    1    1    0    0    1    0    0    0    0    0    0    0    0    0
  167          14.32          -0.04    1    1    2    0    1    0    0    0    0    0    0    0    0    0
  168         -14.10           0.04    1    0   -2    2   -2    0    0    0    0    0    0    0    0    0
  169          10.86           3.18    0    0    1   -1    1    0    0    0   -2    0    0    0    0    0
  170         -10.58          -3.10    0    0    1   -1    0    0    0    0   -2    0    0    0    0    0
  171          -3.62           9.86    0    0    0    0    0    0    0    0    0    1    0    0    0    0
  172         -13.48           0.00    0    0    0    0    0    0    1    1    0    0    0    0    0    2
  173          13.41          -0.04    1    0    2    1    2    0    0    0    0    0    0    0    0    0
  174          13.32          -0.08    2    0    2    0    0    0    0    0    0    0    0    0    0    0
  175         -13.33          -0.04    0    1    0   -2    1    0    0    0    0    0    0    0    0    0
  176         -13.29           0.00    1    0    2   -1    2    0    0    0    0    0    0    0    0    0
  177          -0.20          13.05    0    0    0    0    0    0    3   -4    0    0    0    0    0    0
  178           0.00          13.13    1    0   -1    0    0    0    0    0    0    0    0    0    0    0
  179          -8.99           4.02    1    0    0    0    0    0  -18   16    0    0    0    0    0    0
  180         -12.93           0.04    1    0    0    1    0    0    0    0    0    0    0    0    0    0
  181           2.03          10.82    0    0    0    0    0    0    0    0    0    2    0    0    0    1
  182         -12.78           0.04    0    0    0    1   -1    0    0    0    0    0    0    0    0    0
  183          12.24           0.04    1    0    0   -2    2    0    0    0    0    0    0    0    0    0
  184           8.71           3.54    1    0    0    0    0    0  -10    3    0    0    0    0    0    0
  185          11.98          -0.04    1   -1    0    0   -1    0    0    0    0    0    0    0    0    0
  186         -11.38           0.04    2    1    0    0    0    0    0    0    0    0    0    0    0    0
  187         -11.30           0.00    2    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
  188          11.14          -0.04    0    0    2    1    1    0    0    0    0    0    0    0    0    0
  189          10.98           0.00    0    3    0    0    0    0    0    0    0    0    0    0    0    0
  190         -10.98           0.00    1    2    0   -2    0    0    0    0    0    0    0    0    0    0
  191           0.44         -10.38    0    0    0    0    0    0    5   -8    0    0    0    0    0   -2
  192          10.46           0.08    1    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
  193         -10.42           0.00    0    0    0    0    0    0    0    1    0    1    0    0    0    2
  194         -10.30           0.08    4    0    2    0    2    0    0    0    0    0    0    0    0    0
  195           6.92           3.34    0    0    1   -1    1    0    0   -1    0    0   -1    0    0    0
  196          10.07           0.04    1    0   -2    0    1    0    0    0    0    0    0    0    0    0
  197          10.02           0.00    2    1    2   -2    2    0    0    0    0    0    0    0    0    0
  198          -9.75           0.04    0    1    2    1    2    0    0    0    0    0    0    0    0    0
  199           9.75           0.00    1    1    2   -2    1    0    0    0    0    0    0    0    0    0
  200           9.67          -0.04    1    0    4   -2    2    0    0    0    0    0    0    0    0    0
  201          -1.99           7.72    0    0    1   -1    1    0    0   -1    0   -1    0    0    0    0
  202           0.40           9.27    0    0    2   -2    0    0   -5    6    0    0    0    0    0    0
  203          -3.42           6.09    0    0    0    0    0    0    0    2   -4    0    0    0    0    0
  204           0.56          -8.67    0    0    0    0    0    0    0    0    0    2   -5    0    0    1
  205          -9.19           0.00    2    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  206           9.11           0.00    1    0   -2    1   -1    0    0    0    0    0    0    0    0    0
  207           9.07           0.00    2   -2    0   -2    0    0    0    0    0    0    0    0    0    0
  208           1.63           6.96    0    0    0    0    0    0    8  -13    0    0    0    0    0   -2
  209          -8.47           0.00    0    2    0   -2    0    0    0    0    0    0    0    0    0    0
  210          -8.28           0.04    1    1    0    0   -1    0    0    0    0    0    0    0    0    0
  211           8.27           0.04    2    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
  212          -8.04           0.00    0    0    0    0    0    0    5   -7    0    0    0    0    0   -2
  213           7.91           0.00    0    1    0    2    1    0    0    0    0    0    0    0    0    0
  214          -7.84          -0.04    1    0   -4    0   -1    0    0    0    0    0    0    0    0    0
  215          -7.64           0.08    2    0    2    2    1    0    0    0    0    0    0    0    0    0
  216           5.21          -2.51    1    0    2    0    2    0    0    1    0    0    0    0    0    0
  217          -5.77           1.87    0    0    0    0    0    0    3   -5    0    0    0    0    0    0
  218           5.01          -2.51    1    0   -2    0   -2    0    0    4   -8    3    0    0    0    0
  219          -7.48           0.00    0    0    2   -3    2    0    0    0    0    0    0    0    0    0
  220          -7.32          -0.12    0    0    0    0    0    0    0    4    0   -2    0    0    0    2
  221           7.40          -0.04    0    0    4    0    2    0    0    0    0    0    0    0    0    0
  222           7.44           0.00    1    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  223           6.32          -1.11    1    0    0   -1    1    0    0    0    0    0    0    0    0    0
  224          -6.13          -1.19    0    0    0    0    0    0    0    2    0   -1    0    0    0    2
  225           0.20          -6.88    0    0    0    0    0    0    2   -1    0    0    0    0    0    2
  226           6.92           0.04    1    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  227           6.48          -0.48    0    0    0    0    0    0    0    2    0    1    0    0    0    2
  228          -6.94           0.00    2    0    0   -2   -2    0    0    0    0    0    0    0    0    0
  229           2.47          -4.46    0    0    0    0    0    0    8  -11    0    0    0    0    0   -2
  230          -2.23          -4.65    0    0    0    0    0    0    0    8  -16    4    5    0    0   -2
  231          -1.07          -5.69    0    0    1   -1    1    0    0   -1    0    2    0    0    0    0
  232           4.97          -1.71    0    0    0    0    0    0    0    8  -16    4    5    0    0    2
  233           5.57           1.07    0    0    1   -1    1    0   -5    7    0    0    0    0    0    0
  234          -6.48           0.08    1    0    2    4    2    0    0    0    0    0    0    0    0    0
  235           2.03           4.53    0    0    0    0    0    0    0    0    0    0    2    0    0    1
  236           4.10          -2.39    1    0    0   -2    0    0   19  -21    3    0    0    0    0    0
  237           0.00          -6.44    0    0    1    0    1    0    0    0    0    0    0    0    0    0
  238          -6.40           0.00    3    0    0   -4    0    0    0    0    0    0    0    0    0    0
  239           6.32           0.00    1    1   -2    0   -2    0    0    0    0    0    0    0    0    0
  240           2.67          -3.62    0    0    0    0    0    0    0    3    0   -2    0    0    0    2
  241          -1.91          -4.38    0    0    1   -1    1    0    0   -1    0    0    2    0    0    0
  242          -2.43          -3.82    0    0    0    0    0    0    8  -15    0    0    0    0    0   -2
  243           6.20           0.00    0    1    0   -2   -1    0    0    0    0    0    0    0    0    0
  244          -3.38          -2.78    0    0    0    0    0    0    0    1    2    0    0    0    0    2
  245          -6.12           0.04    0    0    0    4    1    0    0    0    0    0    0    0    0    0
  246          -6.09          -0.04    0    0    2   -4    1    0    0    0    0    0    0    0    0    0
  247          -6.01          -0.04    1    1    0   -4    0    0    0    0    0    0    0    0    0    0
  248           3.18          -2.82    0    0    0    0    0    0    0    3   -2    0    0    0    0    2
  249          -5.05           0.84    0    0    0    0    0    0    0    0    0    3    0    0    0    2
  250           5.85           0.00    3    0    2   -2    1    0    0    0    0    0    0    0    0    0
  251           5.69          -0.12    0    0    0    0    0    0    0    4   -2    0    0    0    0    2
  252           5.73          -0.04    1    1    2    2    2    0    0    0    0    0    0    0    0    0
  253           5.61           0.00    0    0    4   -2    1    0    0    0    0    0    0    0    0    0
  254           5.49           0.00    2    0    0   -2    0    0   -3    3    0    0    0    0    0    0
  255          -5.33           0.04    3    0    2    2    2    0    0    0    0    0    0    0    0    0
  256          -5.29           0.00    0    0    0    0    0    0    0    0    0    0    2    0    0    2
  257           5.25           0.00    2    1    0   -2   -1    0    0    0    0    0    0    0    0    0
  258           0.99           4.22    0    0    0    0    0    0    4   -6    0    0    0    0    0   -1
  259          -0.99           4.22    0    0    0    0    0    0    1   -1    0    0    0    0    0   -1
  260           0.00           5.21    1    0    0   -1    0    0   -3    4    0    0    0    0    0    0
  261           5.13           0.04    0    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  262          -4.90           0.00    2    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  263          -3.10           1.79    0    0    0    0    0    0    0    3   -4    0    0    0    0    0
  264          -4.81           0.04    0    0    2    4    1    0    0    0    0    0    0    0    0    0
  265          -4.75           0.00    0    0    0    2   -2    0    0    0    0    0    0    0    0    0
  266           4.70          -0.04    3    0    0    0   -1    0    0    0    0    0    0    0    0    0
  267          -4.69           0.00    0    0    0    0    0    0    0    1    0   -3    0    0    0   -2
  268          -4.65           0.00    0    0    0    0    0    0    6   -8    0    0    0    0    0   -2
  269           4.65           0.00    0    0    2   -2    1    0   -3    3    0    0    0    0    0    0
  270          -4.57           0.00    2    1    0   -4    0    0    0    0    0    0    0    0    0    0
  271           4.49          -0.04    4    0    2   -2    2    0    0    0    0    0    0    0    0    0
  272          -4.53           0.00    0    0    1   -1    1    0    0    3   -8    3    0    0    0    0
  273           0.00          -4.53    0    0    0    0    0    0    3   -2    0    0    0    0    0    2
  274           0.00          -4.53    0    0    0    0    0    0    2   -5    0    0    0    0    0   -2
  275          -4.53           0.00    2    0    0    2    1    0    0    0    0    0    0    0    0    0
  276           4.50           0.00    1   -1    0   -2   -1    0    0    0    0    0    0    0    0    0
  277          -4.49           0.00    0    0    0    0    0    0    1   -3    0    0    0    0    0   -2
  278           1.83           2.63    0    0    0    0    1    0    8  -13    0    0    0    0    0    0
  279           4.38           0.00    2    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  280           0.88          -3.46    0    0    0    0    0    0    1   -2    0    0    0    0    0    0
  281          -2.70           1.55    0    0    0    0    0    0    0    2   -3    0    0    0    0    0
  282          -4.22           0.00    0    0    2   -1    1    0    0    0    0    0    0    0    0    0
  283          -4.10          -0.12    1    1    0    2    0    0    0    0    0    0    0    0    0    0
  284           3.54          -0.64    2    0    0   -2    1    0    0   -2    0    3    0    0    0    0
  285          -3.50           0.68    0    0    0    0    0    0    5   -8    0    0    0    0    0   -1
  286           4.18           0.00    1   -1   -2    2   -1    0    0    0    0    0    0    0    0    0
  287           4.14           0.00    1    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  288           4.10           0.00    1    0    0   -3    0    0    0    0    0    0    0    0    0    0
  289          -4.06           0.00    2    0    0   -4    1    0    0    0    0    0    0    0    0    0
  290           2.70          -1.35    1    0    0    0   -1    0  -18   16    0    0    0    0    0    0
  291          -4.04           0.00    2    0    0   -2    2    0    0    0    0    0    0    0    0    0
  292          -3.98          -0.04    1    0    0   -4    1    0    0    0    0    0    0    0    0    0
  293          -3.98           0.04    1   -1    2    2    1    0    0    0    0    0    0    0    0    0
  294           4.02           0.00    2    0    2   -4    1    0    0    0    0    0    0    0    0    0
  295           3.94           0.00    0    0    1   -1    1    0    0   -5    8   -3    0    0    0    0
  296           0.84          -3.10    0    0    1   -1    0    0    0   -1    0   -1    0    0    0    0
  297           3.30           0.60    0    0    0    0    0    0    0    2    0   -3    0    0    0    0
  298          -1.59           2.27    0    0    0    0    1    0   -8   13    0    0    0    0    0    0
  299          -3.66          -0.20    2   -1    0   -2    0    0    0    0    0    0    0    0    0    0
  300          -3.10          -0.72    2    0    0   -2    0    0   -6    8    0    0    0    0    0    0
  301          -3.82           0.00    1   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
  302          -3.62          -0.16    2    1   -2    0   -1    0    0    0    0    0    0    0    0    0
  303          -3.74           0.00    0    1   -2    1   -2    0    0    0    0    0    0    0    0    0
  304           3.74           0.00    4    0    0    0    0    0    0    0    0    0    0    0    0    0
  305          -3.74           0.00    0    2   -2    2    0    0    0    0    0    0    0    0    0    0
  306          -3.71           0.00    0    2    0    0    1    0    0    0    0    0    0    0    0    0
  307           3.02           0.68    0    0    0    0    1    0    0    0    0   -2    5    0    0    0
  308           3.70           0.00    1    0   -4    2   -2    0    0    0    0    0    0    0    0    0
  309           3.30           0.40    0    2   -2    0   -2    0    0    0    0    0    0    0    0    0
  310          -3.66           0.04    2   -1    2    2    2    0    0    0    0    0    0    0    0    0
  311           3.66           0.04    0    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  312          -3.62           0.00    1    0   -2   -3   -2    0    0    0    0    0    0    0    0    0
  313          -3.61           0.00    2    0    0    0    2    0    0    0    0    0    0    0    0    0
  314          -2.90           0.68    0    0    0    0    1    0    0    0    0    2   -5    0    0    0
  315           0.80          -2.78    0    0    0    0    1    0    0    0    0    1    0    0    0    0
  316           3.54           0.00    0    0    0    0    0    0    3   -3    0    0    0    0    0    0
  317          -3.54           0.00    0    0    0    0    0    0    0    2    0    0    0    0    0    2
  318          -3.50           0.00    2    0    2   -2    0    0    0    0    0    0    0    0    0    0
  319           3.45           0.00    0    2    2   -2    1    0    0    0    0    0    0    0    0    0
  320           0.00          -3.42    0    0    0    0    0    0    0    6  -16    4    5    0    0   -2
  321           3.38           0.00    1   -2    0    0    0    0    0    0    0    0    0    0    0    0
  322           2.27          -1.11    1    0    0    0    1    0  -18   16    0    0    0    0    0    0
  323          -3.34           0.00    1   -1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  324           3.34           0.00    0    1    2    2    1    0    0    0    0    0    0    0    0    0
  325          -3.30           0.01    0    0    2    0    3    0    0    0    0    0    0    0    0    0
  326           3.31           0.00    1    0    2    0   -1    0    0    0    0    0    0    0    0    0
  327           3.30           0.00    3    0    0    0    1    0    0    0    0    0    0    0    0    0
  328          -3.30           0.00    1    0   -2   -1   -2    0    0    0    0    0    0    0    0    0
  329          -1.39          -1.91    0    0    0    0    0    0    0    1    0    2    0    0    0    2
  330           3.30           0.00    0    0    0    0    0    0    4   -4    0    0    0    0    0    0
  331           3.26           0.00    2    1    0   -2    1    0    0    0    0    0    0    0    0    0
  332           3.26           0.00    1    0    0    4    0    0    0    0    0    0    0    0    0    0
  333           3.22          -0.04    1    0    2    2    0    0    0    0    0    0    0    0    0    0
  334          -3.26           0.00    2   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
  335           2.51          -0.64    0    0    0    0    0    0    3   -7    0    0    0    0    0   -2
  336           3.14           0.00    2    0   -4    0   -2    0    0    0    0    0    0    0    0    0
  337          -2.63          -0.48    0    0    1   -1    1    0    0   -1    0    0    1    0    0    0
  338           3.10           0.00    3    0   -2    0   -1    0    0    0    0    0    0    0    0    0
  339          -3.06           0.00    2   -1    2    0    1    0    0    0    0    0    0    0    0    0
  340           2.94          -0.12    1   -2    0   -2    0    0    0    0    0    0    0    0    0    0
  341           3.06           0.00    2    0   -2    2   -1    0    0    0    0    0    0    0    0    0
  342           0.00           2.98    0    0    1    0    0    0    0    0    0    0    0    0    0    0
  343           2.98           0.00    1    0    2   -4    0    0    0    0    0    0    0    0    0    0
  344           2.07           0.91    0    0    0    0    0    0    0    0    0    1    0    0    0    2
  345          -2.98           0.00    0    0    0    0    0    0    0    1    0    0   -1    0    0    0
  346           2.94           0.00    0    2    2    0    2    0    0    0    0    0    0    0    0    0
  347          -2.94           0.00    0    0    0    0    0    0    7   -9    0    0    0    0    0   -2
  348          -2.94           0.00    0    0    0    0    0    0    0    0    2    0    0    0    0    2
  349          -2.90           0.00    1   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  350          -0.56          -2.35    0    0    0    0    0    0    2   -4    0    0    0    0    0   -1
  351          -1.47           1.39    0    0    0    0    1    0    0    1   -2    0    0    0    0    0
  352           2.80           0.00    1    0    2   -4    2    0    0    0    0    0    0    0    0    0
  353          -2.74           0.00    2    0    2   -1    2    0    0    0    0    0    0    0    0    0
  354          -0.12           2.63    0    0    0    0    0    0    4   -7    0    0    0    0    0   -2
  355           2.15          -0.60    0    0    0    0    0    0    3   -3    0    0    0    0    0    2
  356          -2.70           0.00    1    1   -2    1   -1    0    0    0    0    0    0    0    0    0
  357           1.79          -0.88    0    0    0    0    0    0    0    8  -15    0    0    0    0    0
  358          -0.48           2.19    0    0    0    0    0    0    2   -2    0    0    0    0    0   -1
  359           0.44           2.23    0    0    0    0    0    0    0    0    0    0    1    0    0    0
  360           0.52           2.07    0    0    0    0    0    0    0    0    0    1    0    0    0   -1
  361          -2.59           0.00    0    1    0   -4    0    0    0    0    0    0    0    0    0    0
  362           2.55           0.00    2    1    2    0    1    0    0    0    0    0    0    0    0    0
  363          -1.11           1.43    0    0    0    0    0    0    0    1    0   -3    0    0    0    0
  364          -2.51           0.00    3   -1    2    0    2    0    0    0    0    0    0    0    0    0
  365          -2.51           0.00    2    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  366           2.51           0.00    1    1   -2    0   -1    0    0    0    0    0    0    0    0    0
  367           0.00          -2.50    1    0   -1    0   -3    0    0    0    0    0    0    0    0    0
  368           2.47           0.00    1   -1    0   -2    1    0    0    0    0    0    0    0    0    0
  369           2.11          -0.36    2    0    0   -2   -1    0    0   -2    0    3    0    0    0    0
  370           1.67           0.80    0    0    1   -1    0    0    0   -1    0    0   -1    0    0    0
  371           2.46           0.00    0    2    0    0   -1    0    0    0    0    0    0    0    0    0
  372          -2.43           0.00    1    0   -2    1    0    0    0    0    0    0    0    0    0    0
  373          -2.39           0.00    1    0    2   -3    2    0    0    0    0    0    0    0    0    0
  374          -1.83           0.56    0    0    0    0    0    0    4   -6    0    0    0    0    0    0
  375          -0.44          -1.95    0    0    0    0    0    0    3   -5    0    0    0    0    0   -1
  376           0.24           2.15    0    0    0    0    0    0    0    0    0    1    0    0    0    1
  377           2.39           0.00    2    0   -2   -2    0    0    0    0    0    0    0    0    0    0
  378           2.35           0.00    0    0    0    0    0    0    5   -5    0    0    0    0    0    0
  379           2.27           0.00    1    0    2    1    1    0    0    0    0    0    0    0    0    0
  380          -2.22           0.00    2    0    0    0   -2    0    0    0    0    0    0    0    0    0
  381          -1.03          -1.15    1    0    0   -1    1    0    0   -1    0    2    0    0    0    0
  382           1.87           0.32    0    0    0    0    0    0    0    0    0    0    1    0    0    1
  383          -0.32          -1.87    0    0    1   -1    1    0    0   -2    2    0    0    0    0    0
  384           2.15           0.00    2    0    0   -2    1    0    0   -2    0    2    0    0    0    0
  385          -0.80           1.35    0    0    0    0    0    0    0    3   -5    0    0    0    0    0
  386           2.11           0.00    3    1    2    0    2    0    0    0    0    0    0    0    0    0
  387          -2.11           0.00    1    1    2    1    2    0    0    0    0    0    0    0    0    0
  388          -0.56          -1.55    0    0    0    0    0    0    5   -8    0    0    0    0    0    0
  389           2.11           0.00    1    0    0   -1    0    0    0   -1    0    1    0    0    0    0
  390          -0.84          -1.27    0    0    0    0    0    0    0    0    4    0    0    0    0    2
  391          -1.99           0.12    0    0    0    0    0    0    8  -10    0    0    0    0    0   -2
  392          -0.24           1.87    0    0    0    0    0    0    0    1   -2    0    0    0    0   -1
  393          -0.24          -1.87    0    0    1   -1    1    0    0   -1    0    1    0    0    0    0
  394          -2.03           0.00    1   -2    2    0    2    0    0    0    0    0    0    0    0    0
  395           2.03           0.00    2   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  396           2.03           0.00    2    0    0   -3    0    0    0    0    0    0    0    0    0    0
  397           2.03           0.00    0    0    0    0    0    0    0    2   -4    0    0    0    0   -2
  398          -0.40           1.59    0    0    1   -1    1    0   -3    4    0    0    0    0    0    0
  399           1.99           0.00    0    0    2   -2    1    0    0   -2    0    2    0    0    0    0
  400           1.95           0.00    0    1    0    1    1    0    0    0    0    0    0    0    0    0
  401           1.95           0.00    3    0    0    2    0    0    0    0    0    0    0    0    0    0
  402           1.91           0.00    0    1    0    1   -1    0    0    0    0    0    0    0    0    0
  403           1.19          -0.72    0    0    0    0    0    0    0    5   -4    0    0    0    0    2
  404           1.87           0.00    2    0    2    1    2    0    0    0    0    0    0    0    0    0
  405           1.87           0.00    1    1    0    1    0    0    0    0    0    0    0    0    0    0
  406          -1.27           0.60    0    0    0    0    0    0    0    0    0    0    1    0    0   -1
  407           0.72          -1.15    0    0    0    0    0    0    0    4   -4    0    0    0    0    2
  408          -0.99           0.88    0    0    0    0    0    0    0    0    0    2    0    0    0    0
  409           1.87           0.00    2   -1    0    2    0    0    0    0    0    0    0    0    0    0
  410          -1.87           0.00    1    1   -2    2   -1    0    0    0    0    0    0    0    0    0
  411          -1.83           0.00    2    0   -4   -2   -2    0    0    0    0    0    0    0    0    0
  412          -1.79           0.00    0    1    0    2   -1    0    0    0    0    0    0    0    0    0
  413          -1.79           0.00    4    0    2    0    1    0    0    0    0    0    0    0    0    0
  414           1.79           0.00    1    2    2   -2    2    0    0    0    0    0    0    0    0    0
  415           0.00          -1.79    0    0    0    0    0    0    6   -9    0    0    0    0    0   -2
  416          -1.79           0.00    1    1    2   -4    1    0    0    0    0    0    0    0    0    0
  417          -1.75           0.00    0    0    4   -4    2    0    0    0    0    0    0    0    0    0
  418          -1.75           0.00    3    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
  419           1.75           0.00    2    1   -2    0    0    0    0    0    0    0    0    0    0    0
  420          -1.47          -0.28    0    0    0    0    0    0    0    4    0   -3    0    0    0    2
  421          -1.71           0.00    1    0    2   -3    1    0    0    0    0    0    0    0    0    0
  422           1.71           0.00    2    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  423           0.32           1.39    0    0    0    0    0    0    5   -7    0    0    0    0    0   -1
  424           0.28          -1.43    0    0    0    0    0    0    0    2    0    0    0    0    0    1
  425          -0.52          -1.19    0    0    0    0    0    0    0    1    0    2   -5    0    0    0
  426           1.67           0.00    2    0    0    2   -1    0    0    0    0    0    0    0    0    0
  427          -1.67           0.00    0    1    2    1    1    0    0    0    0    0    0    0    0    0
  428           0.76          -0.91    0    0    0    0    0    0    2    1    0    0    0    0    0    2
  429          -0.32           1.35    0    0    0    0    0    0    3   -3    0    0    0    0    0   -1
  430          -1.39          -0.28    0    0    1   -1    0    0   -5    7    0    0    0    0    0    0
  431           1.63           0.00    2    1    2   -2    1    0    0    0    0    0    0    0    0    0
  432          -1.59           0.00    1    0    2   -1    1    0    0    0    0    0    0    0    0    0
  433           1.03          -0.56    0    0    0    0    0    0    0    4   -3    0    0    0    0    2
  434           1.59           0.00    0    0    0    0    0    0    6   -6    0    0    0    0    0    0
  435           1.55           0.00    1    0    4   -2    1    0    0    0    0    0    0    0    0    0
  436          -0.28          -1.27    0    0    0    0    0    0    1   -1    0    0    0    0    0    1
  437          -0.64           0.91    0    0    0    0    0    0    0    4   -6    0    0    0    0    0
  438          -0.32          -1.23    0    0    0    0    0    0    1   -3    0    0    0    0    0   -1
  439          -1.55           0.00    1   -1    2   -2    1    0    0    0    0    0    0    0    0    0
  440          -1.51           0.00    1    0    2   -2   -1    0    0    0    0    0    0    0    0    0
  441           1.51           0.00    2   -2    0   -2   -1    0    0    0    0    0    0    0    0    0
  442          -1.51           0.00    2   -1    0   -2    1    0    0    0    0    0    0    0    0    0
  443           1.51           0.00    2    0    0   -1    0    0    0    0    0    0    0    0    0    0
  444           1.47           0.00    2    0    4   -2    2    0    0    0    0    0    0    0    0    0
  445           1.47           0.00    0    0    2    3    2    0    0    0    0    0    0    0    0    0
  446           0.95          -0.52    0    0    0    0    0    0    0    3   -6    0    0    0    0   -2
  447           1.23          -0.24    0    0    0    0    0    0    3   -1    0    0    0    0    0    2
  448           0.60           0.88    2    0    0   -2    0    0    0   -6    8    0    0    0    0    0
  449          -1.47           0.00    1   -1    0    2    1    0    0    0    0    0    0    0    0    0
  450          -1.43           0.00    1    0   -2    2    1    0    0    0    0    0    0    0    0    0
  451           1.43           0.00    1   -1    0    2   -1    0    0    0    0    0    0    0    0    0
  452           1.43           0.00    0    1    4   -2    2    0    0    0    0    0    0    0    0    0
  453          -0.68          -0.76    0    0    1   -1    1    0    0   -1    0    0    0    0    2    0
  454           0.95          -0.48    2    0    2    0    2    0    0    2    0   -3    0    0    0    0
  455          -0.95          -0.48    0    0    2    0    2    0    0    1    0    0    0    0    0    0
  456          -1.19          -0.24    0    0    0    0    0    0    0    1    0   -4    0    0    0   -2
  457           0.36          -1.07    0    0    1   -1    1    0    0   -1    0   -4   10    0    0    0
  458           0.95           0.48    2    0    0   -2    0    0    0   -5    6    0    0    0    0    0
  459           1.43           0.00    1    1    0   -1    0    0    0    0    0    0    0    0    0    0
  460           1.39           0.00    0    0    0    4   -1    0    0    0    0    0    0    0    0    0
  461           1.39           0.00    2   -1    0    0    1    0    0    0    0    0    0    0    0    0
  462          -1.39           0.00    1   -2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  463          -1.39           0.00    1    2    0    0    0    0    0    0    0    0    0    0    0    0
  464           0.00           1.39    0    0    0    0    0    0    0    2    0   -1    0    0    0    0
  465          -0.12          -1.27    0    0    0    0    0    0    0    4    0   -1    0    0    0    2
  466           0.56           0.84    0    0    0    0    0    0    0    2   -4    0    0    0    0   -1
  467          -0.44          -0.95    0    0    0    0    0    0    0    0    0    2   -5    0    0   -2
  468           0.32          -1.07    0    0    0    0    0    0    0    1    0    0    0    0    0    0
  469           1.03          -0.36    2    0   -1   -1    0    0    0    3   -7    0    0    0    0    0
  470          -0.28           1.11    0    0    1   -1    1    0   -4    5    0    0    0    0    0    0
  471           0.44           0.95    0    0    1   -1    2    0    0   -1    0    0    2    0    0    0
  472          -1.35           0.00    1   -1   -2    2    0    0    0    0    0    0    0    0    0    0
  473           0.88           0.48    0    0    0    0    0    0    0    1   -8    3    0    0    0   -2
  474          -1.35           0.00    0    0    0    0    0    0    9  -11    0    0    0    0    0   -2
  475           1.35           0.00    1    0    0   -2    0    0    0   -2    0    2    0    0    0    0
  476           1.35           0.00    1    0    0   -1   -2    0    0    0    0    0    0    0    0    0
  477          -1.31           0.00    0    1   -2    2    1    0    0    0    0    0    0    0    0    0
  478           1.31           0.00    1    0   -2    1   -2    0    0    0    0    0    0    0    0    0
  479          -1.19          -0.12    0    0    0    0    0    0    2   -3    0    0    0    0    0   -1
  480           1.27           0.00    0    0    4    0    1    0    0    0    0    0    0    0    0    0
  481           0.40          -0.88    0    0    2   -2    1    0    0   -9   13    0    0    0    0    0
  482           1.27           0.00    1    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  483           1.27           0.00    3    1    2   -2    2    0    0    0    0    0    0    0    0    0
  484          -0.16          -1.11    0    0    0    0    0    0    1    0    0    0    0    0    0    0
  485          -0.84           0.44    2    0    0    0    0    0    0   -2    0    3    0    0    0    0
  486           0.84          -0.44    1    0    2    0    1    0    0   -2    0    3    0    0    0    0
  487           0.84          -0.44    1    0   -2    0   -1    0    0   -1    0    0    0    0    0    0
  488          -1.27           0.00    1    0   -4   -2   -2    0    0    0    0    0    0    0    0    0
  489          -1.27           0.00    1    0    0    4    1    0    0    0    0    0    0    0    0    0
  490           1.27           0.00    1    1    0    2    1    0    0    0    0    0    0    0    0    0
  491          -0.44          -0.84    0    0    0    0    1    0    0   -2    4    0    0    0    0    0
  492           0.00          -1.27    0    0    0    0    1    0    2   -3    0    0    0    0    0    0
  493          -1.27           0.00    2    0    2   -4    2    0    0    0    0    0    0    0    0    0
  494          -1.23           0.00    1   -1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  495          -1.23           0.00    1   -2    2    2    2    0    0    0    0    0    0    0    0    0
  496           1.23           0.00    0    0    2   -1    0    0    0    0    0    0    0    0    0    0
  497           0.00           1.23    0    0    0    0    0    0    0    3   -6    0    0    0    0    0
  498          -0.12           1.11    0    0    1    1    1    0    0    1    0    0    0    0    0    0
  499           1.22           0.00    0    0    2   -4    2    0    0    0    0    0    0    0    0    0
  500           1.19           0.00    1    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  501          -0.24           0.95    0    0    0    0    0    0    4   -4    0    0    0    0    0   -1
  502          -0.76          -0.44    0    0    0    0    0    0    0    3   -8    3    0    0    0    0
  503           0.91           0.28    0    0    0    0    0    0    0    1   -2    0    0    0    0    1
  504           1.19           0.00    1    1   -2    1    0    0    0    0    0    0    0    0    0    0
  505           1.19           0.00    3    0    2    0    0    0    0    0    0    0    0    0    0    0
  506           0.00           1.19    0    0    1    0    2    0    0    0    0    0    0    0    0    0
  507           1.15           0.00    2   -1    0    0   -1    0    0    0    0    0    0    0    0    0
  508           0.00           1.15    1    0    0   -1    1    0   -3    4    0    0    0    0    0    0
  509          -1.15           0.00    1    0    2    4    1    0    0    0    0    0    0    0    0    0
  510           1.15           0.00    2   -1   -2    0    0    0    0    0    0    0    0    0    0    0
  511          -1.15           0.00    2    0   -2    2    0    0    0    0    0    0    0    0    0    0
  512           1.15           0.00    3   -1    0    0    0    0    0    0    0    0    0    0    0    0
  513           1.15           0.00    0    0    0    0    0    0    0    3    0    0   -1    0    0    2
  514          -0.95           0.20    0    0    0    0    0    0    6  -10    0    0    0    0    0   -2
  515           0.24           0.91    0    0    0    0    0    0    0    0    0    3    0    0    0    1
  516          -1.15           0.00    1    1    2   -2    0    0    0    0    0    0    0    0    0    0
  517          -1.12           0.00    0    1    2   -2   -1    0    0    0    0    0    0    0    0    0
  518          -1.11           0.00    1    0    0    1   -1    0    0    0    0    0    0    0    0    0
  519          -1.11           0.00    1    2    0   -2   -1    0    0    0    0    0    0    0    0    0
  520           0.16           0.95    0    0    0    0    1    0    0    0    0    0    1    0    0    0
  521          -1.11           0.00    2    0    2   -4    0    0    0    0    0    0    0    0    0    0
  522           1.11           0.00    0    0    0    0    0    0    7   -7    0    0    0    0    0    0
  523           0.20          -0.91    0    0    0    0    0    0    1    1    0    0    0    0    0    1
  524          -0.72          -0.40    0    0    0    0    0    0    0    0    3    0    0    0    0    2
  525          -1.11           0.00    1    1   -2   -2    0    0    0    0    0    0    0    0    0    0
  526          -1.11           0.00    0    0    0    0    1    0    1   -1    0    0    0    0    0    0
  527           1.07           0.00    2   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  528          -1.07           0.00    2    0    2    4    2    0    0    0    0    0    0    0    0    0
  529           0.76          -0.32    0    0    0    0    0    0    0    1   -4    0    0    0    0   -2
  530           0.00          -1.07    0    0    0    0    0    0    3   -6    0    0    0    0    0   -2
  531           1.07           0.00    0    0    0    0    0    0    0    3   -3    0    0    0    0    0
  532           1.07           0.00    0    0    2   -2    1    0   -4    4    0    0    0    0    0    0
  533          -1.07           0.00    0    1   -2   -2    0    0    0    0    0    0    0    0    0    0
  534           1.07           0.00    0    0    2   -3    1    0    0    0    0    0    0    0    0    0
  535          -0.84          -0.24    0    0    0    0    1    0   -3    5    0    0    0    0    0    0
  536           0.00          -1.03    0    0    3    0    3    0    0    0    0    0    0    0    0    0
  537           1.03           0.00    2    1    2    2    2    0    0    0    0    0    0    0    0    0
  538          -1.03           0.00    3    0    2   -4    2    0    0    0    0    0    0    0    0    0
  539          -0.24           0.80    0    0    0    0    0    0    0    1    0    0   -2    0    0    0
  540           0.20           0.84    0    0    1   -1    1    0   -1    0    0    0    0    0    0    0
  541          -1.03           0.00    3    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  542          -1.03           0.00    0    0    0    0    3    0    0    0    0    0    0    0    0    0
  543          -0.99           0.00    2    0    0    1    0    0    0    0    0    0    0    0    0    0
  544           0.24           0.76    0    0    0    0    0    0    2   -1    0    0    0    0    0    0
  545          -0.99           0.00    0    0    0    0    0    0    0    7   -8    3    0    0    0    2
  546          -0.16           0.84    0    0    0    0    0    0    0    2    0   -2    0    0    0    1
  547          -0.99           0.00    0    0    0    0    0    0    0    1   -1    0    0    0    0    0
  548          -0.64           0.36    0    0    0    0    0    0    0    4   -7    0    0    0    0   -2
  549           0.99           0.00    1   -2    0    2    0    0    0    0    0    0    0    0    0    0
  550           0.36          -0.64    0    0    1   -1    0    0    0   -1    0   -2    5    0    0    0
  551          -0.95           0.00    3    0    2    2    1    0    0    0    0    0    0    0    0    0
  552          -0.95           0.00    1   -1    2    4    2    0    0    0    0    0    0    0    0    0
  553           0.00           0.95    0    0    1   -1    0    0    0   -1    0   -1    1    0    0    0
  554           0.64           0.32    1    0    0   -1    0    0    0   -3    4    0    0    0    0    0
  555           0.00          -0.95    0    0    0    0    0    0    7  -10    0    0    0    0    0   -2
  556           0.84           0.12    0    0    0    0    0    0    0    5   -8    3    0    0    0    0
  557           0.20           0.76    0    0    0    0    0    0    6   -8    0    0    0    0    0   -1
  558          -0.95           0.00    3    0   -2    0    0    0    0    0    0    0    0    0    0    0
  559           0.95           0.00    1    0    4    0    2    0    0    0    0    0    0    0    0    0
  560          -0.95           0.00    1   -1   -2    2   -2    0    0    0    0    0    0    0    0    0
  561           0.00           0.92    1    0    0   -1   -1    0   -3    4    0    0    0    0    0    0
  562           0.91           0.00    1    1    2    2    1    0    0    0    0    0    0    0    0    0
  563           0.91           0.00    3    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  564           0.40           0.52    1   -2    2   -2    1    0    0    0    0    0    0    0    0    0
  565          -0.91           0.00    5    0    2    0    2    0    0    0    0    0    0    0    0    0
  566          -0.56           0.36    0    0    0    0    0    0    3   -9    4    0    0    0    0   -2
  567           0.44          -0.48    0    0    1   -1    1    0    8  -14    0    0    0    0    0    0
  568          -0.91           0.00    2    0    0   -6    0    0    0    0    0    0    0    0    0    0
  569          -0.91           0.00    3    1    0    0    0    0    0    0    0    0    0    0    0    0
  570          -0.36          -0.56    0    0    1   -1    1    0    3   -6    0    0    0    0    0    0
  571           0.91           0.00    1    0    0    2    2    0    0    0    0    0    0    0    0    0
  572          -0.88           0.00    1    2    0   -2    1    0    0    0    0    0    0    0    0    0
  573          -0.88           0.00    2    1    0    0   -1    0    0    0    0    0    0    0    0    0
  574           0.60          -0.28    0    0    0    0    1    0    0    8  -15    0    0    0    0    0
  575           0.88           0.00    0    2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  576           0.36          -0.52    0    0    1   -1    1    0    0   -3    4    0    0    0    0    0
  577          -0.52           0.36    0    0    0    0    0    0    3   -5    4    0    0    0    0    2
  578           0.52           0.36    0    0    0    0    0    0    0    6    0    0    0    0    0    2
  579           0.00           0.88    0    0    0    0    0    0    0    4   -7    0    0    0    0    0
  580           0.56           0.32    0    0    0    0    0    0    0    8  -15    0    0    0    0   -2
  581           0.64          -0.24    0    0    0    0    0    0    0    4   -5    0    0    0    0    0
  582           0.88           0.00    1   -1    0   -4    0    0    0    0    0    0    0    0    0    0
  583           0.88           0.00    1   -1    2    0    0    0    0    0    0    0    0    0    0    0
  584           0.88           0.00    1   -1    2   -1    2    0    0    0    0    0    0    0    0    0
  585           0.84           0.00    0    1    0   -2   -2    0    0    0    0    0    0    0    0    0
  586          -0.68          -0.16    2    0    0   -2    1    0   -6    8    0    0    0    0    0    0
  587           0.84           0.00    1    2    2    0    2    0    0    0    0    0    0    0    0    0
  588           0.56           0.28    0    0    1   -1    0    0   -8   12    0    0    0    0    0    0
  589          -0.16           0.68    0    0    0    0    0    0    5   -5    0    0    0    0    0   -1
  590           0.64          -0.20    0    0    0    0    0    0    5   -7    0    0    0    0    0    0
  591           0.16           0.68    0    0    0    0    0    0    1    2    0    0    0    0    0    2
  592           0.72          -0.12    0    0    0    0    0    0    0    3    0   -3    0    0    0    0
  593          -0.83           0.00    2    0   -2    0    2    0    0    0    0    0    0    0    0    0
  594          -0.80           0.00    2    1    0    0    1    0    0    0    0    0    0    0    0    0
  595           0.80           0.00    2    2    0   -2    0    0    0    0    0    0    0    0    0    0
  596          -0.80           0.00    4    0    0   -2    0    0    0    0    0    0    0    0    0    0
  597           0.28           0.52    0    0    0    0    0    0    0    5   -9    0    0    0    0    0
  598           0.68          -0.12    0    0    0    0    0    0    3    0    0    0    0    0    0    2
  599           0.00          -0.80    0    0    0    0    0    0    0    2    0    0   -1    0    0    2
  600          -0.32           0.48    0    0    0    0    0    0    0    5   -9    0    0    0    0   -2
  601           0.36          -0.44    0    0    0    0    0    0    0    3    0   -3    0    0    0    2
  602          -0.36          -0.44    0    0    0    0    0    0    0    0    0    0    0    0    2    1
  603          -0.80           0.00    0    1    2    0    0    0    0    0    0    0    0    0    0    0
  604           0.79           0.00    1    1    0   -2   -2    0    0    0    0    0    0    0    0    0
  605           0.74          -0.04    0    0    2   -2    3    0    0    0    0    0    0    0    0    0
  606          -0.76           0.00    0    0    0    0    1    0   -1    1    0    0    0    0    0    0
  607           0.00           0.76    0    0    0    0    1    0   -2    3    0    0    0    0    0    0
  608           0.16           0.60    0    0    0    0    1    0    0    0    0   -1    0    0    0    0
  609          -0.76           0.00    1    0    0    1    1    0    0    0    0    0    0    0    0    0
  610          -0.76           0.00    3    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  611           0.76           0.00    2    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  612          -0.76           0.00    2   -1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  613           0.76           0.00    1    0   -4    2   -1    0    0    0    0    0    0    0    0    0
  614           0.12           0.64    0    0    1   -1    0    0    0   -1    0    2    0    0    0    0
  615           0.76           0.00    1    0    0   -2    0    0   -3    3    0    0    0    0    0    0
  616           0.00           0.76    0    0    1   -1    1    0    0   -9   15    0    0    0    0    0
  617           0.76           0.00    0    0    0    0    0    0    8   -8    0    0    0    0    0    0
  618           0.64          -0.12    0    0    0    0    0    0    7  -11    0    0    0    0    0   -2
  619           0.16          -0.60    0    0    0    0    0    0    2    0    0    0    0    0    0    1
  620           0.76           0.00    0    0    0    0    0    0    0    3   -1    0    0    0    0    2
  621           0.00          -0.76    0    0    0    0    0    0    0    2    0    0    1    0    0    2
  622           0.28          -0.48    0    0    0    0    0    0    0    5   -5    0    0    0    0    2
  623           0.32           0.44    0    0    0    0    0    0    0    1    0    1    0    0    0    0
  624          -0.76           0.00    1    1    2    0    0    0    0    0    0    0    0    0    0    0
  625           0.72           0.00    4    0    2   -2    1    0    0    0    0    0    0    0    0    0
  626           0.72           0.00    1   -1   -2    0    0    0    0    0    0    0    0    0    0    0
  627           0.48          -0.24    0    0    0    0    0    0    0    2    0   -2    5    0    0    2
  628          -0.72           0.00    0    1    2   -4    1    0    0    0    0    0    0    0    0    0
  629           0.72           0.00    0    0    1   -1    1    0   -2    2    0    0    0    0    0    0
  630          -0.72           0.00    0    0    2    1    0    0    0    0    0    0    0    0    0    0
  631          -0.72           0.00    3    0   -2    0   -2    0    0    0    0    0    0    0    0    0
  632          -0.71           0.00    1   -1    0    0    2    0    0    0    0    0    0    0    0    0
  633          -0.68           0.00    1    0   -2   -2    1    0    0    0    0    0    0    0    0    0
  634          -0.68           0.00    0    2    0   -2    1    0    0    0    0    0    0    0    0    0
  635           0.68           0.00    0    1   -4    2   -2    0    0    0    0    0    0    0    0    0
  636           0.68           0.00    0    1    2    4    2    0    0    0    0    0    0    0    0    0
  637           0.68           0.00    1    0   -2    4   -1    0    0    0    0    0    0    0    0    0
  638          -0.68           0.00    0    2    2   -2    0    0    0    0    0    0    0    0    0    0
  639           0.56          -0.12    0    0    0    0    0    0    5   -3    0    0    0    0    0    2
  640          -0.68           0.00    0    0    0    0    0    0    0    3   -5    0    0    0    0   -2
  641          -0.68           0.00    0    0    0    0    0    0    0    2    0   -4    0    0    0   -2
  642           0.20           0.48    0    0    0    0    0    0    0    3    0    2   -5    0    0    2
  643          -0.44          -0.24    0    0    0    0    0    0    0    2    0    2   -5    0    0    2
  644          -0.68           0.00    1    1   -2    0    0    0    0    0    0    0    0    0    0    0
  645           0.64           0.00    2    0    0   -2    1    0   -3    3    0    0    0    0    0    0
  646           0.64           0.00    3    0    0   -2   -1    0    0    0    0    0    0    0    0    0
  647          -0.64           0.00    2   -1    2    2    1    0    0    0    0    0    0    0    0    0
  648           0.64           0.00    0    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  649          -0.64           0.00    1    0   -2   -3   -1    0    0    0    0    0    0    0    0    0
  650          -0.12           0.52    0    0    0    0    0    0    6   -6    0    0    0    0    0   -1
  651          -0.12          -0.52    0    0    0    0    0    0    2   -2    0    0    0    0    0    1
  652          -0.16          -0.48    0    0    0    0    0    0    0    3    0   -2    0    0    0    0
  653          -0.20          -0.44    0    0    0    0    0    0    2   -3    0    0    0    0    0   -2
  654          -0.44           0.20    0    0    0    0    0    0    0   11    0    0    0    0    0    2
  655          -0.44           0.20    0    0    0    0    0    0    0    6  -15    0    0    0    0   -2
  656           0.24          -0.40    0    0    0    0    0    0    0    4   -8    0    0    0    0   -2
  657          -0.20          -0.44    0    0    0    0    0    0    0    3    0    1    0    0    0    2
  658          -0.64           0.00    1    0    0   -6    0    0    0    0    0    0    0    0    0    0
  659           0.40          -0.24    0    0    0    0    1    0    3   -7    4    0    0    0    0    0
  660          -0.64           0.00    1    0   -2   -4    0    0    0    0    0    0    0    0    0    0
  661           0.64           0.00    0    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  662          -0.63           0.00    2    0   -2    2   -2    0    0    0    0    0    0    0    0    0
  663          -0.60           0.00    0    1   -2    1   -1    0    0    0    0    0    0    0    0    0
  664           0.00           0.60    0    0    2   -2    2    0   -8   11    0    0    0    0    0    0
  665          -0.60           0.00    2    0   -2   -3   -2    0    0    0    0    0    0    0    0    0
  666          -0.60           0.00    4    0    2    2    2    0    0    0    0    0    0    0    0    0
  667           0.60           0.00    0    0    0    0    0    0    0    2    0    0   -2    0    0    0
  668           0.00           0.60    0    0    0    0    0    0    0    0    0    0    3    0    0    2
  669           0.24          -0.36    2    0    0   -2    0    0    0   -2    0    4   -3    0    0    0
  670           0.12           0.48    0    0    0    0    0    0    7   -9    0    0    0    0    0   -1
  671           0.48          -0.12    0    0    0    0    0    0    4   -7    0    0    0    0    0   -1
  672           0.12           0.48    0    0    0    0    0    0    3   -3    0    0    0    0    0    1
  673           0.24          -0.36    0    0    0    0    0    0    0    6   -6    0    0    0    0    2
  674           0.36           0.24    0    0    0    0    0    0    0    6  -11    0    0    0    0    0
  675           0.12           0.48    0    0    0    0    0    0    0    5    0   -2    0    0    0    2
  676           0.44           0.16    0    0    0    0    0    0    0    2    0   -4    0    0    0    0
  677          -0.60           0.00    0    0    0    3    0    0    0    0    0    0    0    0    0    0
  678          -0.60           0.00    2    0    0   -2    0    0    0   -2    0    3   -1    0    0    0
  679           0.60           0.00    2   -1    0   -4    0    0    0    0    0    0    0    0    0    0
  680           0.00           0.60    0    0    1   -1    2    0    0   -2    2    0    0    0    0    0
  681           0.59           0.00    0    0    0    0    1    0    0    1    0   -1    0    0    0    0
  682          -0.56           0.00    3    0    0   -4    1    0    0    0    0    0    0    0    0    0
  683          -0.44          -0.12    2    0    0   -2   -1    0   -6    8    0    0    0    0    0    0
  684           0.56           0.00    1    2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  685           0.00           0.56    0    0    1   -1    1    0    0   -1    0   -1    1    0    0    0
  686          -0.56           0.00    3    1    0   -4    0    0    0    0    0    0    0    0    0    0
  687          -0.56           0.00    3    0    0   -6    0    0    0    0    0    0    0    0    0    0
  688           0.56           0.00    0    0    0    0    0    0    0    6    0    0    0    0    0    0
  689          -0.56           0.00    0    0    0    0    0    0    0    1    0    0    1    0    0    2
  690           0.16           0.40    0    0    0    0    0    1    0   -4    0    0    0    0    0   -2
  691           0.44          -0.12    0    0    0    0    0    0    4   -4    0    0    0    0    0    2
  692           0.20          -0.36    0    0    0    0    0    0    0    1   -5    0    0    0    0   -2
  693          -0.36          -0.20    0    0    0    0    1    0   -3    7   -4    0    0    0    0    0
  694          -0.56           0.00    0    0    4   -4    1    0    0    0    0    0    0    0    0    0
  695           0.55           0.00    1    1    0   -2    2    0    0    0    0    0    0    0    0    0
  696           0.52           0.00    1    0    0   -4   -2    0    0    0    0    0    0    0    0    0
  697          -0.52           0.00    1    1    0   -4    1    0    0    0    0    0    0    0    0    0
  698           0.52           0.00    0    0    0    0    1    0    3   -5    0    2    0    0    0    0
  699           0.52           0.00    0    1    0   -1    0    0    0    0    0    0    0    0    0    0
  700           0.16           0.36    0    0    1   -1    0    0    0   -1    0    0    2    0    0    0
  701          -0.52           0.00    0    0    2   -2    0    0   -3    3    0    0    0    0    0    0
  702          -0.52           0.00    1    0    0   -2    0    0    0   -2    0    3    0    0    0    0
  703          -0.52           0.00    1    0   -2   -2   -2    0    0   -2    0    2    0    0    0    0
  704          -0.52           0.00    0    0    2    6    2    0    0    0    0    0    0    0    0    0
  705           0.00          -0.52    0    0    1   -1    1    0    2   -4    0   -3    0    0    0    0
  706           0.52           0.00    0    0    0    0    0    0    9   -9    0    0    0    0    0    0
  707          -0.52           0.00    0    0    0    0    0    0    0    1    1    0    0    0    0    2
  708           0.12           0.40    0    0    0    0    0    0    5   -6    0    0    0    0    0    2
  709           0.52           0.00    1    0   -4    0    0    0    0    0    0    0    0    0    0    0
  710          -0.52           0.00    2    0   -2   -4    0    0    0    0    0    0    0    0    0    0
  711           0.00          -0.52    1    0    1    0    1    0    0    0    0    0    0    0    0    0
  712           0.52           0.00    1    1    0    0    2    0    0    0    0    0    0    0    0    0
  713           0.52           0.00    0    0    0    1    2    0    0    0    0    0    0    0    0    0
  714          -0.51           0.00    1    0    0    2   -2    0    0    0    0    0    0    0    0    0
  715          -0.51           0.00    1    0    2    0    3    0    0    0    0    0    0    0    0    0
  716           0.48           0.00    0    1    0    2    2    0    0    0    0    0    0    0    0    0
  717           0.48           0.00    0    0    2    2   -1    0    0    0    0    0    0    0    0    0
  718          -0.16           0.32    0    0    0    0    1    0    0    2   -4    0    0    0    0    0
  719          -0.48           0.00    1   -2    2   -2    2    0    0    0    0    0    0    0    0    0
  720          -0.48           0.00    0    1   -2   -1   -2    0    0    0    0    0    0    0    0    0
  721           0.48           0.00    3   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  722           0.48           0.00    4    0    0   -4    0    0    0    0    0    0    0    0    0    0
  723          -0.48           0.00    3   -1    2    2    2    0    0    0    0    0    0    0    0    0
  724          -0.12          -0.36    0    0    1   -1    1    0    0   -1    0    3    0    0    0    0
  725          -0.32           0.16    0    0    1   -1    1    0    0   -1    0    0    3    0    0    0
  726           0.32          -0.16    0    0    0    0    0    0    4   -2    0    0    0    0    0    2
  727          -0.12          -0.36    0    0    0    0    0    0    3   -5    0    0    0    0    0    1
  728           0.16           0.32    0    0    0    0    0    0    0    7  -13    0    0    0    0   -2
  729           0.20          -0.28    0    0    0    0    0    0    0    5   -7    0    0    0    0    0
  730          -0.20          -0.28    0    0    0    0    0    0    0    1    0    3    0    0    0    2
  731          -0.36           0.12    0    0    0    0    0    0    0    0    0    4    0    0    0    2
  732          -0.48           0.00    0    0    0    0    0    0    0    0    0    2   -5    0    0    2
  733           0.32          -0.16    0    0    0    0    0    0    0    0    0    0    3    0    0    1
  734           0.48           0.00    1   -1    0    4    0    0    0    0    0    0    0    0    0    0
  735          -0.48           0.00    2    1    0    2    0    0    0    0    0    0    0    0    0    0
  736          -0.48           0.00    2    0    0   -2   -1    0    0   -2    0    0    5    0    0    0
  737          -0.48           0.00    3    0    0    2    1    0    0    0    0    0    0    0    0    0
  738           0.00          -0.48    1    0    1   -2    1    0    0    0    0    0    0    0    0    0
  739           0.48           0.00    1    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  740          -0.48           0.00    0    0    2   -2    2    0   -3    3    0    0    0    0    0    0
  741          -0.48           0.00    1    0    0   -1    2    0    0    0    0    0    0    0    0    0
  742           0.00           0.48    0    0    2   -2    2    0   -5    6    0    0    0    0    0    0
  743           0.44           0.00    0    0    0    0    1    0    0   -1    0    1    0    0    0    0
  744          -0.32          -0.12    0    0    1   -1   -1    0    0    0   -2    0    0    0    0    0
  745          -0.44           0.00    0    1   -2    0    0    0    0    0    0    0    0    0    0    0
  746           0.20          -0.24    0    0    0    0    0    0    0    9  -17    0    0    0    0    0
  747           0.44           0.00    3   -1   -2   -1   -2    0    0    0    0    0    0    0    0    0
  748          -0.44           0.00    1   -1    0   -3    0    0    0    0    0    0    0    0    0    0
  749           0.44           0.00    0    0    2   -2    1    0   -2    2    0    0    0    0    0    0
  750           0.20          -0.24    0    0    0    0    0    0    8  -13    0    0    0    0    0    1
  751           0.12           0.32    0    0    0    0    0    0    5  -10    0    0    0    0    0   -2
  752          -0.20           0.24    0    0    0    0    0    0    1    0    0    0    0    0    0    2
  753           0.32          -0.12    0    0    0    0    0    0    1   -3    0    0    0    0    0    0
  754           0.00           0.44    0    0    0    0    0    0    0    6  -11    0    0    0    0   -2
  755           0.00           0.44    0    0    0    0    0    0    0    5   -8    0    0    0    0    0
  756           0.44           0.00    0    0    0    0    0    0    0    1   -3    0    0    0    0   -2
  757          -0.44           0.00    1    2    0   -4    0    0    0    0    0    0    0    0    0    0
  758          -0.44           0.00    0    1    0    4    0    0    0    0    0    0    0    0    0    0
  759          -0.44           0.00    3    0    2   -2    0    0    0    0    0    0    0    0    0    0
  760           0.44           0.00    1   -1    2   -4    1    0    0    0    0    0    0    0    0    0
  761           0.44           0.00    2    0    2    2    0    0    0    0    0    0    0    0    0    0
  762           0.40           0.00    2    0    2    0   -1    0    0    0    0    0    0    0    0    0
  763          -0.40           0.00    2    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  764          -0.40           0.00    2    1    0   -4    1    0    0    0    0    0    0    0    0    0
  765          -0.40           0.00    1    0    4   -4    2    0    0    0    0    0    0    0    0    0
  766           0.40           0.00    2    0   -4    0   -1    0    0    0    0    0    0    0    0    0
  767           0.24           0.16    2    0    0   -2   -1    0    0   -5    6    0    0    0    0    0
  768           0.00          -0.40    0    0    3    0    2    0    0    0    0    0    0    0    0    0
  769           0.12           0.28    2    0   -1   -1   -1    0    0   -1    0    3    0    0    0    0
  770           0.40           0.00    1    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  771          -0.40           0.00    0    0    4   -1    2    0    0    0    0    0    0    0    0    0
  772           0.40           0.00    0    0    0    0    0    0    0    7  -13    0    0    0    0    0
  773           0.40           0.00    5    0    2   -2    2    0    0    0    0    0    0    0    0    0
  774           0.00          -0.40    0    0    0    0    0    0    9  -12    0    0    0    0    0   -2
  775          -0.40           0.00    0    0    0    0    0    0    5   -9    0    0    0    0    0   -2
  776           0.00          -0.40    0    0    0    0    0    0    4   -4    0    0    0    0    0    1
  777           0.00          -0.40    0    0    0    0    0    0    3   -1    0    0    0    0    0    1
  778           0.20          -0.20    0    0    0    0    0    0    2   -4    0    0    0    0    0    0
  779          -0.40           0.00    0    0    0    0    0    0    1   -2    0    0    0    0    0    1
  780          -0.40           0.00    0    0    0    0    0    0    0    5   -3    0    0    0    0    2
  781          -0.12          -0.28    0    0    0    0    0    0    0    4   -8    1    5    0    0   -2
  782           0.40           0.00    0    0    0    0    0    0    3   -5    0    2    0    0    0    0
  783           0.40           0.00    1    1   -2    2    0    0    0    0    0    0    0    0    0    0
  784           0.40           0.00    2    0    0    4    0    0    0    0    0    0    0    0    0    0
  785           0.40           0.00    0    0    0    3    1    0    0    0    0    0    0    0    0    0
  786           0.40           0.00    4    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  787           0.00           0.40    0    0    1   -1    2    0    0   -1    0    2    0    0    0    0
  788          -0.20          -0.16    0    0    0    0    2    0    0   -1    2    0    0    0    0    0
  789           0.36           0.00    1    0    0   -3    1    0    0    0    0    0    0    0    0    0
  790           0.36           0.00    1   -2    0    0    1    0    0    0    0    0    0    0    0    0
  791           0.24          -0.12    0    2   -2    0   -1    0    0    0    0    0    0    0    0    0
  792           0.20          -0.16    2    0   -1   -1   -1    0    0    3   -7    0    0    0    0    0
  793           0.00           0.36    0    0    2    0    2    0    0   -4    8   -3    0    0    0    0
  794           0.00           0.36    0    0    2    0    2    0    0    4   -8    3    0    0    0    0
  795          -0.36           0.00    2    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  796           0.12           0.24    1    0    1    0    0    0    0    0    0    0    0    0    0    0
  797          -0.36           0.00    2   -2    2    0    2    0    0    0    0    0    0    0    0    0
  798          -0.36           0.00    0    3    0   -2    0    0    0    0    0    0    0    0    0    0
  799          -0.36           0.00    2   -1    0   -1    0    0    0    0    0    0    0    0    0    0
  800          -0.36           0.00    1    3    0   -2    0    0    0    0    0    0    0    0    0    0
  801           0.36           0.00    1   -1   -2   -3   -2    0    0    0    0    0    0    0    0    0
  802           0.00           0.36    0    0    1   -1    1    0   -2    3    0    0    0    0    0    0
  803           0.00           0.36    0    0    0    0    0    0    7   -7    0    0    0    0    0   -1
  804           0.00           0.36    0    0    0    0    0    0    6   -7    0    0    0    0    0    0
  805          -0.36           0.00    0    0    0    0    0    0    6   -9    0    0    0    0    0   -1
  806           0.00           0.36    0    0    0    0    0    0    4   -3    0    0    0    0    0    2
  807           0.12          -0.24    0    0    0    0    0    0    1   -2    0    0    0    0    0   -2
  808          -0.24           0.12    0    0    0    0    0    0    0    6   -5    0    0    0    0    2
  809          -0.36           0.00    0    0    0    0    0    0    0    5    0   -3    0    0    0    2
  810           0.00           0.36    0    0    0    0    0    0    0    3   -2    0    0    0    0    0
  811           0.36           0.00    0    0    0    0    0    0    0    3   -6    0    0    0    0   -1
  812           0.24          -0.12    0    0    0    0    0    0    0    2   -5    0    0    0    0   -2
  813           0.00          -0.36    0    0    1   -1    1    0    0    1    0    0    0    0    0    0
  814          -0.36           0.00    1   -2   -2    0   -2    0    0    0    0    0    0    0    0    0
  815           0.36           0.00    2    0    0    2    2    0    0    0    0    0    0    0    0    0
  816           0.36           0.00    0    1    0   -2    2    0    0    0    0    0    0    0    0    0
  817          -0.36           0.00    0    0    2    0   -2    0    0    0    0    0    0    0    0    0
  818           0.36           0.00    0    0    2   -2   -2    0    0    0    0    0    0    0    0    0
  819          -0.13           0.22    0    0    1   -1    2    0    0   -1    0   -2    5    0    0    0
  820          -0.32           0.00    0    0    0    0    1    0    3   -5    0    0    0    0    0    0
  821          -0.32           0.00    1    1    0    2   -1    0    0    0    0    0    0    0    0    0
  822           0.32           0.00    4    0    0    0   -1    0    0    0    0    0    0    0    0    0
  823          -0.20          -0.12    0    0    0    0    1    0    0   -8   15    0    0    0    0    0
  824           0.32           0.00    0    2    2    0    1    0    0    0    0    0    0    0    0    0
  825           0.12           0.20    2    0    0   -2    1    0    0   -6    8    0    0    0    0    0
  826          -0.32           0.00    3   -1    2    0    1    0    0    0    0    0    0    0    0    0
  827           0.32           0.00    0    0    2    0    2    0    1   -1    0    0    0    0    0    0
  828          -0.32           0.00    0    0    2    0    2    0   -1    1    0    0    0    0    0    0
  829          -0.32           0.00    2    0    2   -1    1    0    0    0    0    0    0    0    0    0
  830           0.00          -0.32    2    0   -1   -1   -2    0    0   -1    0    2    0    0    0    0
  831           0.32           0.00    1    2    2   -2    1    0    0    0    0    0    0    0    0    0
  832           0.32           0.00    2    2    2   -2    2    0    0    0    0    0    0    0    0    0
  833           0.12          -0.20    0    0    2   -2    0    0    0   -9   13    0    0    0    0    0
  834          -0.32           0.00    3    0    2   -1    2    0    0    0    0    0    0    0    0    0
  835           0.00          -0.32    1    0    0   -2    0    0   20  -21    0    0    0    0    0    0
  836           0.32           0.00    0    0    2   -2    1    0    0   -2    0    0    2    0    0    0
  837           0.00           0.32    0    0    2   -2    1    0    0   -8   11    0    0    0    0    0
  838           0.00          -0.32    0    0    1   -1    1    0    0   -1    0    0    0    2    0    0
  839           0.00          -0.32    0    0    1   -1    1    0    0   -1    0   -1    2    0    0    0
  840           0.32           0.00    0    0    0    0    0    0    8  -12    0    0    0    0    0    0
  841          -0.32           0.00    0    0    0    0    0    0    5   -5    0    0    0    0    0    2
  842           0.00           0.32    0    0    0    0    0    0    5   -6    0    0    0    0    0    0
  843           0.32           0.00    0    0    0    0    0    0    2   -6    0    0    0    0    0   -2
  844           0.00           0.32    0    0    0    0    0    0    0    8  -15    0    0    0    0   -1
  845           0.00          -0.32    0    0    0    0    0    0    0    5   -2    0    0    0    0    2
  846           0.32           0.00    0    0    0    0    0    0    0    4   -4    0    0    0    0    0
  847          -0.16           0.16    0    0    0    0    0    0    0    2    2    0    0    0    0    2
  848          -0.16           0.16    0    0    0    0    0    0    0    1    0   -4    0    0    0    0
  849           0.00           0.32    0    0    0    0    0    0    0    0    0    0    1    0    0    2
  850           0.20           0.12    0    0    1   -1    1    0    0   -1    0   -2    4    0    0    0
  851           0.20           0.12    0    0    0    0    0    1    0   -4    0    0    0    0    0    0
  852          -0.20           0.12    0    0    0    0    0    0    0    5   -8    0    0    0    0   -2
  853           0.12           0.20    0    0    0    0    0    0    0    4   -8    0    0    0    0    0
  854           0.12          -0.20    0    0    0    0    0    0    0    2   -6    0    0    0    0   -2
  855           0.00           0.32    0    0    2   -2    1   -1    0    2    0    0    0    0    0    0
  856          -0.32           0.00    0    0    4   -2    0    0    0    0    0    0    0    0    0    0
  857           0.32           0.00    0    0    0    1   -2    0    0    0    0    0    0    0    0    0
  858           0.28           0.00    2    0   -4    2   -2    0    0    0    0    0    0    0    0    0
  859          -0.28           0.00    0    0    0    4    2    0    0    0    0    0    0    0    0    0
  860           0.28           0.00    1    0    0    4   -1    0    0    0    0    0    0    0    0    0
  861           0.28           0.00    1   -2    0   -2   -1    0    0    0    0    0    0    0    0    0
  862           0.28           0.00    1    1    0   -1    1    0    0    0    0    0    0    0    0    0
  863           0.16           0.12    1    0    0   -1    1    0    0   -3    4    0    0    0    0    0
  864           0.28           0.00    3    1    2    0    1    0    0    0    0    0    0    0    0    0
  865          -0.28           0.00    1    1    2    1    1    0    0    0    0    0    0    0    0    0
  866          -0.12          -0.16    0    0    0    0    1    0    0   -9   17    0    0    0    0    0
  867           0.28           0.00    1    1    4   -2    2    0    0    0    0    0    0    0    0    0
  868          -0.28           0.00    4    0    2   -4    2    0    0    0    0    0    0    0    0    0
  869          -0.28           0.00    0    2    0    2    0    0    0    0    0    0    0    0    0    0
  870           0.28           0.00    1    0    2   -3    0    0    0    0    0    0    0    0    0    0
  871           0.28           0.00    3    0   -2   -2    0    0    0    0    0    0    0    0    0    0
  872           0.28           0.00    2   -2    0    0    0    0    0    0    0    0    0    0    0    0
  873           0.28           0.00    1    1    0   -3    0    0    0    0    0    0    0    0    0    0
  874           0.28           0.00    1    0    2    3    2    0    0    0    0    0    0    0    0    0
  875           0.28           0.00    1    0    0   -2    0    0    0    1    0   -1    0    0    0    0
  876          -0.28           0.00    1    0   -2   -2   -2    0   -3    3    0    0    0    0    0    0
  877           0.28           0.00    0    0    2   -2    1    0    0   -2    0    3    0    0    0    0
  878           0.00           0.28    0    0    0    0    0    0    8   -8    0    0    0    0    0   -1
  879           0.00           0.28    0    0    0    0    0    0    8  -10    0    0    0    0    0   -1
  880           0.00          -0.28    0    0    0    0    0    0    4   -2    0    0    0    0    0    1
  881          -0.28           0.00    0    0    0    0    0    0    3   -4    0    0    0    0    0   -1
  882           0.28           0.00    0    0    0    0    0    0    3   -6    0    0    0    0    0   -1
  883          -0.12          -0.16    0    0    0    0    0    0    1   -4    0    0    0    0    0   -2
  884           0.00           0.28    0    0    0    0    0    0    0    6    0    0    0    0    0    1
  885           0.00          -0.28    0    0    0    0    0    0    0    6   -7    0    0    0    0    2
  886           0.12          -0.16    0    0    0    0    0    0    0    4    0    0    0    0    0    0
  887          -0.28           0.00    0    0    0    0    0    0    0    4    0    0   -2    0    0    2
  888           0.00          -0.28    0    0    0    0    0    0    0    3    0    0   -2    0    0    2
  889           0.00           0.28    0    0    0    0    0    0    0    1    0   -1    0    0    0    1
  890           0.00          -0.28    0    0    0    0    0    0    0    1   -6    0    0    0    0   -2
  891           0.28           0.00    0    0    0    0    0    0    0    0    0    4   -5    0    0    2
  892          -0.28           0.00    0    0    0    0    0    0    0    0    0    0    0    2    0    2
  893          -0.28           0.00    0    0    0    0    0    0    3   -7    4    0    0    0    0    0
  894           0.28           0.00    1   -1    2   -2    0    0    0    0    0    0    0    0    0    0
  895           0.28           0.00    1    1    2   -4    0    0    0    0    0    0    0    0    0    0
  896           0.12          -0.16    0    0    1   -1    1    0    0   -1    0    0   -2    0    0    0
  897           0.28           0.00    0    0    2    4    0    0    0    0    0    0    0    0    0    0
  898           0.28           0.00    2   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
  899          -0.28           0.00    0    0    1   -1    2    0    0    0   -2    0    0    0    0    0
  900           0.00          -0.28    0    0    0    0    2    0    0    4   -8    3    0    0    0    0
  901           0.00          -0.28    0    0    0    0    2    0    0   -4    8   -3    0    0    0    0
  902           0.28           0.00    1    1    2   -4    2    0    0    0    0    0    0    0    0    0
  903           0.00           0.24    0    0    2   -2   -1    0   -5    6    0    0    0    0    0    0
  904           0.24           0.00    1   -2    0   -2    1    0    0    0    0    0    0    0    0    0
  905          -0.24           0.00    0    1    0   -4    1    0    0    0    0    0    0    0    0    0
  906          -0.24           0.00    1   -2    2    0    1    0    0    0    0    0    0    0    0    0
  907           0.24           0.00    0    0    2    0    2    0    0   -1    0    1    0    0    0    0
  908          -0.24           0.00    0    0    2    0    2    0    0    1    0   -1    0    0    0    0
  909          -0.24           0.00    2    0   -4   -2   -1    0    0    0    0    0    0    0    0    0
  910           0.24           0.00    2    0    2    1    1    0    0    0    0    0    0    0    0    0
  911           0.24           0.00    2    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
  912           0.00          -0.24    2    0   -1   -1   -1    0    0   -1    0    2    0    0    0    0
  913          -0.24           0.00    4   -1    2    0    2    0    0    0    0    0    0    0    0    0
  914           0.24           0.00    5    0    0    0    0    0    0    0    0    0    0    0    0    0
  915          -0.24           0.00    3    0    0   -3    0    0    0    0    0    0    0    0    0    0
  916          -0.24           0.00    2    2    0   -4    0    0    0    0    0    0    0    0    0    0
  917           0.24           0.00    2    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  918          -0.24           0.00    2    1    2    1    2    0    0    0    0    0    0    0    0    0
  919          -0.24           0.00    1    0    2   -2    2    0    0   -2    0    2    0    0    0    0
  920          -0.24           0.00    1    0    0    0    0    0    0   -2    0    2    0    0    0    0
  921           0.24           0.00    1    0    0   -2    0    0    2   -2    0    0    0    0    0    0
  922           0.00          -0.24    1    0   -1    1   -1    0  -18   17    0    0    0    0    0    0
  923           0.24           0.00    0    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  924          -0.24           0.00    0    1    2    3    2    0    0    0    0    0    0    0    0    0
  925           0.24           0.00    0    0    0    0    0    0    8  -12    0    0    0    0    0   -2
  926           0.24           0.00    0    0    0    0    0    0    8  -16    0    0    0    0    0   -2
  927           0.00           0.24    0    0    0    0    0    0    7   -8    0    0    0    0    0    0
  928           0.12          -0.12    0    0    0    0    0    0    2   -3    0    0    0    0    0    1
  929           0.00          -0.24    0    0    0    0    0    0    0    5   -6    0    0    0    0    2
  930          -0.24           0.00    0    0    0    0    0    0    0    4   -6    0    0    0    0   -2
  931           0.00           0.24    0    0    0    0    0    0    0    4   -8    1    5    0    0    2
  932          -0.24           0.00    0    0    0    0    0    0    0    2    0   -2    0    0    0    2
  933           0.00          -0.24    0    0    0    0    0    0    0    2   -7    0    0    0    0   -2
  934           0.24           0.00    0    0    0    0    0    0    0    0    2    0    0    0    0    0
  935          -0.24           0.00    0    0    0    0    0    0    0    0    0    0    5    0    0    2
  936          -0.24           0.00    0    0    0    0    0    0    0    0    0    0    0    0    2    2
  937           0.24           0.00    0    1    0    3    0    0    0    0    0    0    0    0    0    0
  938           0.24           0.00    0    2    2    2    2    0    0    0    0    0    0    0    0    0
  939           0.24           0.00    2   -1    2   -1    2    0    0    0    0    0    0    0    0    0
  940           0.24           0.00    1   -1    2    2    0    0    0    0    0    0    0    0    0    0
  941           0.24           0.00    2    1    2   -4    1    0    0    0    0    0    0    0    0    0
  942          -0.24           0.00    2    0    0   -2   -2    0   -3    3    0    0    0    0    0    0
  943          -0.24           0.00    1   -1    0   -4   -1    0    0    0    0    0    0    0    0    0
  944           0.00           0.24    0    0    1   -1    2    0    0   -1    0    1    0    0    0    0
  945           0.00          -0.24    0    0    1   -1    2    0    0   -1    0   -1    0    0    0    0
  946           0.24           0.00    0    0    0    0    1    0    2   -2    0    0    0    0    0    0
  947           0.00           0.20    1    0   -1    0    1    0    0    0    0    0    0    0    0    0
  948           0.20           0.00    2   -2    0   -2    1    0    0    0    0    0    0    0    0    0
  949          -0.20           0.00    0    0    1   -1    2    0    0   -1    0    0    1    0    0    0
  950           0.20           0.00    2    0    0   -2   -1    0   -3    3    0    0    0    0    0    0
  951           0.20           0.00    1   -2    0    0   -1    0    0    0    0    0    0    0    0    0
  952           0.00           0.20    0    0    0    0    1    0    0    0    0    0   -1    0    0    0
  953          -0.20           0.00    1    0   -2    1    1    0    0    0    0    0    0    0    0    0
  954           0.20           0.00    2    0    0   -3   -1    0    0    0    0    0    0    0    0    0
  955           0.20           0.00    2    1   -2    0    1    0    0    0    0    0    0    0    0    0
  956           0.20           0.00    2    0    0   -1    1    0    0    0    0    0    0    0    0    0
  957           0.20           0.00    2   -1    2   -2    1    0    0    0    0    0    0    0    0    0
  958           0.20           0.00    1   -1   -4    2   -2    0    0    0    0    0    0    0    0    0
  959          -0.20           0.00    1   -2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  960           0.20           0.00    2    0    4   -2    1    0    0    0    0    0    0    0    0    0
  961           0.20           0.00    0    0    2    3    1    0    0    0    0    0    0    0    0    0
  962           0.20           0.00    0    1    4   -2    1    0    0    0    0    0    0    0    0    0
  963           0.20           0.00    1    0   -2    0   -2    0  -10    3    0    0    0    0    0    0
  964           0.00           0.20    1    0    1   -2    0    0    0    0    0    0    0    0    0    0
  965           0.20           0.00    1   -1    2    1    2    0    0    0    0    0    0    0    0    0
  966           0.20           0.00    4    1    2    0    2    0    0    0    0    0    0    0    0    0
  967           0.20           0.00    0    0    2   -2    1    0    0   -1    0    1    0    0    0    0
  968           0.20           0.00    1    0    0   -1    0    0    0   -2    2    0    0    0    0    0
  969          -0.20           0.00    0    0    2   -2    0    0    0   -2    0    2    0    0    0    0
  970           0.00          -0.20    1    0    0    0    0    0    0    4   -8    3    0    0    0    0
  971          -0.20           0.00    4   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  972           0.20           0.00    2    1   -2   -6   -2    0    0    0    0    0    0    0    0    0
  973          -0.20           0.00    2    0    0   -2    0    0    0   -4    8   -3    0    0    0    0
  974           0.20           0.00    2    0    0   -2    0    0   -2    2    0    0    0    0    0    0
  975           0.20           0.00    2    0    0   -2   -1    0    0   -2    0    4   -5    0    0    0
  976           0.00          -0.20    2    0   -1   -1   -1    0    0   -1    0    0    0    0    0    0
  977          -0.20           0.00    1    1   -2   -3   -2    0    0    0    0    0    0    0    0    0
  978           0.00          -0.20    1    0    3    0    3    0    0    0    0    0    0    0    0    0
  979           0.00           0.20    1    0    1   -1    1    0    0   -1    0    0    0    0    0    0
  980           0.00          -0.20    1    0    0    0    0    0    0   -4    8   -3    0    0    0    0
  981           0.20           0.00    1    0    0    0    0    0   -1    1    0    0    0    0    0    0
  982           0.00           0.20    1    0    0   -2    0    0   17  -16    0   -2    0    0    0    0
  983          -0.20           0.00    1    0   -1   -1   -1    0   20  -20    0    0    0    0    0    0
  984           0.20           0.00    1    0   -2   -2   -2    0    0   -2    0    3    0    0    0    0
  985           0.20           0.00    0    3   -2   -2   -2    0    0    0    0    0    0    0    0    0
  986           0.00           0.20    0    0    1   -1    1    0    1   -2    0    0    0    0    0    0
  987           0.00           0.20    0    0    1   -1    1    0   -2    1    0    0    0    0    0    0
  988          -0.20           0.00    0    0    1   -1    0    0    0    0    0    0    0    0    0    0
  989           0.00           0.20    0    0    0    0    0    0    9   -9    0    0    0    0    0   -1
  990           0.00           0.20    0    0    0    0    0    0    9  -11    0    0    0    0    0   -1
  991           0.00           0.20    0    0    0    0    0    0    6  -10    0    0    0    0    0   -1
  992           0.00          -0.20    0    0    0    0    0    0    5   -3    0    0    0    0    0    1
  993          -0.20           0.00    0    0    0    0    0    0    4   -5    0    0    0    0    0   -1
  994           0.00          -0.20    0    0    0    0    0    0    3   -4    0    0    0    0    0   -2
  995           0.00          -0.20    0    0    0    0    0    0    0    6   -9    0    0    0    0    0
  996           0.00          -0.20    0    0    0    0    0    0    0    5  -10    0    0    0    0   -2
  997          -0.20           0.00    0    0    0    0    0    0    0    4    0   -4    0    0    0    2
  998           0.20           0.00    0    0    0    0    0    0    0    3    0   -4    0    0    0    0
  999          -0.20           0.00    0    0    0    0    0    0    0    2    0    0    0    0    0    0
 1000          -0.20           0.00    0    0    0    0    0    0    0    2    0   -5    0    0    0   -2
 1001           0.00           0.20    0    0    0    0    0    0    0    1    0   -2    5    0    0    2
 1002           0.20           0.00    0    0    0    0    0    0    0    1    0   -2    0    0    0   -2
 1003           0.00          -0.20    0    0    0    0    0    0    0    1    0   -3    0    0    0   -1
 1004          -0.20           0.00    0    0    0    0    0    0    0    1    0   -5    0    0    0   -2
 1005           0.00          -0.20    0    0    0    0    0    0    0    0    0    0    0    2    0    1
 1006           0.00           0.20    0    0    0    0    0    0    0    0    0    0    0    1   -2   -2
 1007          -0.20           0.00    0    2    0   -4    0    0    0    0    0    0    0    0    0    0
 1008           0.20           0.00    0    0    0    6    0    0    0    0    0    0    0    0    0    0
 1009          -0.20           0.00    3    0    0   -2    0    0    0    0    0    0    0    0    0    0
 1010           0.20           0.00    2   -1   -2    2   -1    0    0    0    0    0    0    0    0    0
 1011           0.00          -0.20    1    0   -1    1   -1    0    0    1    0    0    0    0    0    0
 1012          -0.20           0.00    0    0    2   -4    0    0    0    0    0    0    0    0    0    0
 1013          -0.20           0.00    1    0    0   -1   -1    0    0   -2    2    0    0    0    0    0
 1014           0.20           0.00    0    1    0    4    1    0    0    0    0    0    0    0    0    0
 1015           0.20           0.00    1    0    0   -6   -1    0    0    0    0    0    0    0    0    0
 1016           0.00           0.20    0    0    0    0    1    0    5   -8    0    0    0    0    0    0
 1017           0.20           0.00    0    0    4    2    2    0    0    0    0    0    0    0    0    0
 1018           0.20           0.00    3    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1019          -0.20           0.00    0    0    2   -2    2    0    0   -2    0    2    0    0    0    0
 1020          -0.20           0.00    1   -1    0   -2   -2    0    0    0    0    0    0    0    0    0
 1021          -0.20           0.00    0    0    1   -1    2    0   -5    7    0    0    0    0    0    0
 1022          -0.20           0.00    2    1    0   -2   -2    0    0    0    0    0    0    0    0    0
 1023           0.20           0.00    0    0    0    0    1    0    0    2   -2    0    0    0    0    0
 1024          -0.20           0.00    1    0   -2    0    2    0    0    0    0    0    0    0    0    0
 1025          -0.20           0.00    1   -1    0    0   -2    0    0    0    0    0    0    0    0    0
 1026           0.20           0.00    1    1   -2    2   -2    0    0    0    0    0    0    0    0    0
 1027           0.16           0.00    0    0    0    0    1    0   -2    2    0    0    0    0    0    0
 1028           0.16           0.00    1    0    0   -3   -1    0    0    0    0    0    0    0    0    0
 1029           0.00          -0.16    0    0    1   -1   -1    0    0   -1    0   -1    0    0    0    0
 1030           0.16           0.00    2    0   -2   -2    1    0    0    0    0    0    0    0    0    0
 1031           0.16           0.00    1    0    0   -1    1    0    0   -1    0    1    0    0    0    0
 1032           0.16           0.00    2    0    0   -3    1    0    0    0    0    0    0    0    0    0
 1033           0.16           0.00    3    0    0    2   -1    0    0    0    0    0    0    0    0    0
 1034           0.16           0.00    1    1    0    1    1    0    0    0    0    0    0    0    0    0
 1035          -0.16           0.00    0    2    0    0   -2    0    0    0    0    0    0    0    0    0
 1036           0.16           0.00    2   -1    0    2   -1    0    0    0    0    0    0    0    0    0
 1037           0.16           0.00    1    1    0    1   -1    0    0    0    0    0    0    0    0    0
 1038          -0.16           0.00    1    0   -4   -2   -1    0    0    0    0    0    0    0    0    0
 1039          -0.16           0.00    1   -1   -2   -4   -1    0    0    0    0    0    0    0    0    0
 1040          -0.16           0.00    1   -2    2    2    1    0    0    0    0    0    0    0    0    0
 1041           0.16           0.00    3    1    2   -2    1    0    0    0    0    0    0    0    0    0
 1042           0.16           0.00    1    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
 1043          -0.16           0.00    2    0    4   -4    2    0    0    0    0    0    0    0    0    0
 1044          -0.04           0.12    2   -2    2   -2    2    0    0    0    0    0    0    0    0    0
 1045           0.16           0.00    3    0    4   -2    2    0    0    0    0    0    0    0    0    0
 1046           0.00           0.16    0    0    1   -1    0    0    3   -6    0    0    0    0    0    0
 1047           0.16           0.00    1    1    2    4    2    0    0    0    0    0    0    0    0    0
 1048           0.16           0.00    0    0    2   -3    0    0    0    0    0    0    0    0    0    0
 1049           0.00          -0.16    0    0    1    1    0    0    0    1    0    0    0    0    0    0
 1050           0.00          -0.16    0    0    1   -1    0    0   -4    5    0    0    0    0    0    0
 1051           0.00           0.16    0    0    1   -1    0    0    0   -2    2    0    0    0    0    0
 1052           0.16           0.00    0    0    1   -1    0    0    0   -1    0    0    1    0    0    0
 1053           0.00           0.16    0    0    1   -1    0    0    0   -1    0    1    0    0    0    0
 1054          -0.16           0.00    0    1   -2    4   -1    0    0    0    0    0    0    0    0    0
 1055           0.00          -0.16    1    0    0   -2    0    0    0    4   -8    3    0    0    0    0
 1056           0.16           0.00    3    0    2    1    2    0    0    0    0    0    0    0    0    0
 1057           0.16           0.00    3    0    0   -1    0    0    0    0    0    0    0    0    0    0
 1058           0.16           0.00    2    1    0    1    0    0    0    0    0    0    0    0    0    0
 1059           0.16           0.00    2    0    2    0    1    0    0    1    0    0    0    0    0    0
 1060          -0.16           0.00    2    0    0   -2    0    0    0   -2    0    2    2    0    0    0
 1061           0.16           0.00    2    0    0   -2    0    0    0   -4    4    0    0    0    0    0
 1062          -0.16           0.00    2   -2    2    2    2    0    0    0    0    0    0    0    0    0
 1063           0.16           0.00    1    1   -2   -6   -2    0    0    0    0    0    0    0    0    0
 1064          -0.16           0.00    1    0    0    0    0    0    1   -1    0    0    0    0    0    0
 1065          -0.16           0.00    1    0    0    0    0    0    0   -1    0    1    0    0    0    0
 1066          -0.16           0.00    1    0    0    0    0    0   -3    3    0    0    0    0    0    0
 1067          -0.16           0.00    1    0    0   -2    0    0    1   -1    0    0    0    0    0    0
 1068           0.00          -0.16    1    0    0   -2    0    0    0   -4    8   -3    0    0    0    0
 1069          -0.16           0.00    1    0    0   -2    0    0   -2    2    0    0    0    0    0    0
 1070          -0.16           0.00    0    0    2   -2    1    0    0    1    0   -1    0    0    0    0
 1071           0.16           0.00    0    0    2   -2    1    0    0   -3    0    3    0    0    0    0
 1072           0.16           0.00    0    0    2   -2    1    0   -5    5    0    0    0    0    0    0
 1073          -0.16           0.00    0    0    1   -1    1    0    1   -3    0    0    0    0    0    0
 1074           0.00           0.16    0    0    1   -1    1    0    0   -1    0    0    0   -1    0    0
 1075           0.16           0.00    0    0    1   -1    1    0    0   -4    6    0    0    0    0    0
 1076           0.00           0.16    0    0    1   -1    1    0   -5    6    0    0    0    0    0    0
 1077          -0.16           0.00    0    0    0    2    0    0    0   -1    0    1    0    0    0    0
 1078           0.00           0.16    0    0    0    0    0    0    8   -9    0    0    0    0    0    0
 1079          -0.16           0.00    0    0    0    0    0    0    7  -10    0    0    0    0    0   -1
 1080           0.00          -0.16    0    0    0    0    0    0    5   -5    0    0    0    0    0    1
 1081           0.00          -0.16    0    0    0    0    0    0    4   -5    0    0    0    0    0   -2
 1082           0.00          -0.16    0    0    0    0    0    0    3   -6    0    0    0    0    0    0
 1083           0.00           0.16    0    0    0    0    0    0    3   -8    0    0    0    0    0   -2
 1084           0.16           0.00    0    0    0    0    0    0    2   -5    0    0    0    0    0   -1
 1085          -0.16           0.00    0    0    0    0    0    0    1    1    0    0    0    0    0    0
 1086           0.16           0.00    0    0    0    0    0    0    1   -2    0    0    0    0    0   -1
 1087           0.00          -0.16    0    0    0    0    0    0    0    7   -8    0    0    0    0    2
 1088           0.00          -0.16    0    0    0    0    0    0    0    7   -9    0    0    0    0    2
 1089           0.00           0.16    0    0    0    0    0    0    0    6  -10    0    0    0    0   -2
 1090           0.00           0.16    0    0    0    0    0    0    0    3    0    0    0    0    0    2
 1091           0.16           0.00    0    0    0    0    0    0    0    3   -8    3    0    0    0   -2
 1092           0.00           0.16    0    0    0    0    0    0    0    2    0    0   -2    0    0    1
 1093           0.16           0.00    0    0    0    0    0    0    0    2   -4    0    0    0    0    1
 1094           0.16           0.00    0    0    0    0    0    0    0    1    0    0    0    0    0   -1
 1095           0.00          -0.16    0    0    0    0    0    0    0    1    0   -1    0    0    0   -1
 1096           0.00          -0.16    0    0    0    0    0    0    0    0    0    3   -5    0    0    0
 1097          -0.16           0.00    0    0    0    0    0    0    0    0    0    2   -2    0    0    0
 1098          -0.16           0.00    2    1    0   -6    0    0    0    0    0    0    0    0    0    0
 1099          -0.16           0.00    1   -1    0    1    0    0    0    0    0    0    0    0    0    0
 1100           0.00          -0.16    0    0    1   -1    0    0   -3    4    0    0    0    0    0    0
 1101           0.16           0.00    3   -1    0    2    0    0    0    0    0    0    0    0    0    0
 1102           0.16           0.00    4    0    0    2    0    0    0    0    0    0    0    0    0    0
 1103          -0.16           0.00    0    0    2   -2    0    0   -4    4    0    0    0    0    0    0
 1104           0.16           0.00    1    0    2   -1    0    0    0    0    0    0    0    0    0    0
 1105           0.16           0.00    1   -2   -2    2   -1    0    0    0    0    0    0    0    0    0
 1106           0.16           0.00    1   -1   -2   -2    0    0    0    0    0    0    0    0    0    0
 1107          -0.16           0.00    2    0    2   -3    2    0    0    0    0    0    0    0    0    0
 1108          -0.16           0.00    0    1    2    2    0    0    0    0    0    0    0    0    0    0
 1109           0.00           0.16    0    0    2   -2    1    0    0   -4    8   -3    0    0    0    0
 1110           0.00           0.16    0    0    2   -2    1    0    0    4   -8    3    0    0    0    0
 1111           0.16           0.00    3    0   -2    2   -1    0    0    0    0    0    0    0    0    0
 1112           0.16           0.00    2   -1    2    0    0    0    0    0    0    0    0    0    0    0
 1113          -0.16           0.00    2    0    0    4    1    0    0    0    0    0    0    0    0    0
 1114           0.16           0.00    2    0   -4    2   -1    0    0    0    0    0    0    0    0    0
 1115          -0.16           0.00    2   -1    0    2    1    0    0    0    0    0    0    0    0    0
 1116          -0.16           0.00    2   -2    0   -2   -2    0    0    0    0    0    0    0    0    0
 1117           0.00          -0.16    0    0    0    0    1    0    0    1    0   -2    0    0    0    0
 1118          -0.16           0.00    3    0    0    0    2    0    0    0    0    0    0    0    0    0
 1119           0.00          -0.16    0    0    0    0    1    0    3   -4    0    0    0    0    0    0
 1120           0.16           0.00    0    2    0    0    2    0    0    0    0    0    0    0    0    0
 1121           0.16           0.00    0    1    2   -4    2    0    0    0    0    0    0    0    0    0
 1122          -0.15           0.00    0    0    1   -1    2    0   -8   12    0    0    0    0    0    0
 1123           0.00           0.12    0    0    0    0    1    0    0   -1    0    2    0    0    0    0
 1124           0.12           0.00    0    0    0    0    1    0    0   -2    2    0    0    0    0    0
 1125           0.00           0.12    0    0    0    0    1    0   -3    4    0    0    0    0    0    0
 1126           0.12           0.00    1    0    0    0   -1    0  -10    3    0    0    0    0    0    0
 1127           0.12           0.00    1    0    0    0    1    0  -10    3    0    0    0    0    0    0
 1128           0.00          -0.12    0    0    0    0    2    0    0    0    0    1    0    0    0    0
 1129          -0.12           0.00    0    1    4   -4    2    0    0    0    0    0    0    0    0    0
 1130          -0.12           0.00    2    0    2   -2   -1    0    0    0    0    0    0    0    0    0
 1131           0.12           0.00    1    0    2   -4   -1    0    0    0    0    0    0    0    0    0
 1132          -0.12           0.00    1    2    0    0    1    0    0    0    0    0    0    0    0    0
 1133           0.00           0.12    2    0    0   -2   -1    0    0   -6    8    0    0    0    0    0
 1134           0.12           0.00    2   -1   -2    0    1    0    0    0    0    0    0    0    0    0
 1135           0.12           0.00    2    0    0   -2    1    0    0   -5    6    0    0    0    0    0
 1136           0.12           0.00    1    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
 1137           0.12           0.00    2    0   -1   -1    1    0    0    3   -7    0    0    0    0    0
 1138           0.12           0.00    1    0    0   -2    1    0    0   -2    0    2    0    0    0    0
 1139           0.12           0.00    1    0    0   -1   -1    0    0   -3    4    0    0    0    0    0
 1140           0.12           0.00    0    0    0    3    2    0    0    0    0    0    0    0    0    0
 1141           0.12           0.00    2    2    0   -2   -1    0    0    0    0    0    0    0    0    0
 1142           0.12           0.00    1   -1    2   -3    1    0    0    0    0    0    0    0    0    0
 1143           0.12           0.00    1    0    4    0    1    0    0    0    0    0    0    0    0    0
 1144          -0.12           0.00    2    0    0   -2   -1    0    0   -2    0    3   -1    0    0    0
 1145           0.12           0.00    2   -1   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1146           0.12           0.00    2    1    2    2    1    0    0    0    0    0    0    0    0    0
 1147           0.00           0.12    0    0    2    0    2    0    2   -3    0    0    0    0    0    0
 1148           0.00           0.12    0    0    2    0    2    0   -2    3    0    0    0    0    0    0
 1149          -0.12           0.00    0    0    2    0    2    0    2   -2    0    0    0    0    0    0
 1150           0.12           0.00    0    0    2    0    2    0   -2    2    0    0    0    0    0    0
 1151          -0.12           0.00    5    0    2    0    1    0    0    0    0    0    0    0    0    0
 1152           0.12           0.00    3    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
 1153           0.12           0.00    1    2    2    0    1    0    0    0    0    0    0    0    0    0
 1154          -0.12           0.00    1   -1    2    4    1    0    0    0    0    0    0    0    0    0
 1155           0.12           0.00    2    0   -2   -2   -2    0    0   -2    0    2    0    0    0    0
 1156          -0.12           0.00    2    0    2    4    1    0    0    0    0    0    0    0    0    0
 1157           0.00           0.12    0    0    2   -2    1    0   -8   11    0    0    0    0    0    0
 1158           0.00           0.12    1    0    2    0    2    0    0   -4    8   -3    0    0    0    0
 1159           0.00           0.12    1    0    2    0    2    0    0    4   -8    3    0    0    0    0
 1160          -0.12           0.00    1   -2    0   -1    0    0    0    0    0    0    0    0    0    0
 1161           0.12           0.00    2    0    2   -2    2    0    0   -2    0    3    0    0    0    0
 1162           0.12           0.00    1    0    2    0    2    0    1   -1    0    0    0    0    0    0
 1163          -0.12           0.00    1    0    2    0    2    0   -1    1    0    0    0    0    0    0
 1164           0.00          -0.12    0    0    2   -2    0   -1    0    2    0    0    0    0    0    0
 1165          -0.12           0.00    4    0   -2   -2    0    0    0    0    0    0    0    0    0    0
 1166          -0.12           0.00    1    0    2   -6    1    0    0    0    0    0    0    0    0    0
 1167           0.12           0.00    0    1   -4    2   -1    0    0    0    0    0    0    0    0    0
 1168           0.00           0.12    0    0    1   -1    0    0    0   -1    0    0   -2    0    0    0
 1169           0.12           0.00    1   -1   -2    1   -1    0    0    0    0    0    0    0    0    0
 1170           0.00           0.12    0    0    1   -1    0    0    0    1    0    0    0    0    0    0
 1171           0.00           0.12    2    0   -1   -1    0    0    0   -1    0    3    0    0    0    0
 1172           0.00          -0.12    0    0    1   -1    0    0   -1    0    0    0    0    0    0    0
 1173           0.08           0.04    0    0    0    0    0    0    0    0    1    0    0    0    0    0
 1174           0.12           0.00    4    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1175           0.12           0.00    3    1    2    2    2    0    0    0    0    0    0    0    0    0
 1176          -0.12           0.00    3    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1177           0.12           0.00    3    1   -2   -6   -2    0    0    0    0    0    0    0    0    0
 1178          -0.12           0.00    3   -1    0   -2    0    0    0    0    0    0    0    0    0    0
 1179          -0.12           0.00    3   -1    0   -3    0    0    0    0    0    0    0    0    0    0
 1180           0.12           0.00    2    1    0   -3    0    0    0    0    0    0    0    0    0    0
 1181           0.00           0.12    2    0    1   -3    1    0   -6    7    0    0    0    0    0    0
 1182           0.00          -0.12    2    0    0   -2    0    0    2   -5    0    0    0    0    0    0
 1183           0.00           0.12    2    0    0   -2    0    0    0   -2    0    5   -5    0    0    0
 1184           0.12           0.00    2    0    0   -2    0    0    0   -2    0    1    5    0    0    0
 1185           0.12           0.00    2    0    0   -2    0    0    0   -2    0    0    2    0    0    0
 1186           0.12           0.00    2    0    0   -2    0    0   -4    4    0    0    0    0    0    0
 1187           0.00           0.12    2    0   -2    0   -2    0    0    5   -9    0    0    0    0    0
 1188          -0.12           0.00    2    0   -2   -5   -2    0    0    0    0    0    0    0    0    0
 1189          -0.12           0.00    2   -1    2    4    2    0    0    0    0    0    0    0    0    0
 1190           0.12           0.00    1    3   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1191          -0.12           0.00    1    1   -2   -1   -2    0    0    0    0    0    0    0    0    0
 1192          -0.12           0.00    1    0    2   -2    2    0   -3    3    0    0    0    0    0    0
 1193           0.00           0.12    1    0    1    1    1    0    0    1    0    0    0    0    0    0
 1194           0.12           0.00    1    0    0    0    0    0    0    1    0   -1    0    0    0    0
 1195           0.12           0.00    1    0    0    0    0    0    0   -2    0    3    0    0    0    0
 1196          -0.12           0.00    1    0    0   -2    0    0    0    2    0   -2    0    0    0    0
 1197           0.12           0.00    1    0   -1    0   -1    0   -3    5    0    0    0    0    0    0
 1198          -0.12           0.00    1    0   -1   -1    0    0    0    8  -15    0    0    0    0    0
 1199           0.00          -0.12    1    0   -1   -2   -1    0    0    0    0    0    0    0    0    0
 1200          -0.12           0.00    1    0   -2   -2   -2    0    0    1    0   -1    0    0    0    0
 1201           0.12           0.00    0    0    2    2    2    0    0    2    0   -2    0    0    0    0
 1202          -0.12           0.00    0    0    2   -2    1    0    0   -2    0    1    0    0    0    0
 1203          -0.12           0.00    0    0    2   -2    1    0    0   -2    0    0    0    0    0    0
 1204           0.12           0.00    0    0    2   -2    1    0    0   -4    4    0    0    0    0    0
 1205           0.00           0.12    0    0    2   -2    1    0    0   -7    9    0    0    0    0    0
 1206          -0.12           0.00    0    0    2   -2    1    0    0  -10   15    0    0    0    0    0
 1207           0.12           0.00    0    0    1   -1    1    0    0    1   -4    0    0    0    0    0
 1208           0.00           0.12    0    0    1   -1    1    0    0   -1    0    1   -3    0    0    0
 1209           0.00           0.12    0    0    1   -1    1    0   -1    2    0    0    0    0    0    0
 1210           0.12           0.00    0    0    1   -1    1    0   -4    6    0    0    0    0    0    0
 1211          -0.12           0.00    0    0    0    2    0    0    0    2    0   -2    0    0    0    0
 1212          -0.12           0.00    0    0    0    2    0    0   -2    2    0    0    0    0    0    0
 1213           0.12           0.00    0    0    0    0    0    0    9  -13    0    0    0    0    0   -2
 1214          -0.12           0.00    0    0    0    0    0    0    8  -11    0    0    0    0    0   -1
 1215           0.12           0.00    0    0    0    0    0    0    8  -14    0    0    0    0    0   -2
 1216           0.00          -0.12    0    0    0    0    0    0    7  -11    0    0    0    0    0   -1
 1217           0.00          -0.12    0    0    0    0    0    0    6   -4    0    0    0    0    0    1
 1218           0.00          -0.12    0    0    0    0    0    0    6   -6    0    0    0    0    0    1
 1219          -0.12           0.00    0    0    0    0    0    0    6   -7    0    0    0    0    0   -1
 1220          -0.12           0.00    0    0    0    0    0    0    6   -8    0    0    0    0    0    0
 1221           0.00          -0.12    0    0    0    0    0    0    6   -9    0    0    0    0    0    0
 1222           0.00           0.12    0    0    0    0    0    0    5   -4    0    0    0    0    0    2
 1223          -0.12           0.00    0    0    0    0    0    0    5   -6    0    0    0    0    0   -1
 1224           0.00          -0.12    0    0    0    0    0    0    5   -6    0    0    0    0    0   -2
 1225          -0.12           0.00    0    0    0    0    0    0    5   -6   -4    0    0    0    0   -2
 1226           0.00           0.12    0    0    0    0    0    0    4   -2    0    0    0    0    0    0
 1227           0.00          -0.12    0    0    0    0    0    0    4   -5    0    0    0    0    0    0
 1228           0.12           0.00    0    0    0    0    0    0    4   -8    0    0    0    0    0   -2
 1229          -0.12           0.00    0    0    0    0    0    0    3   -1    0    0    0    0    0    0
 1230          -0.12           0.00    0    0    0    0    0    0    3   -3    0    2    0    0    0    2
 1231           0.12           0.00    0    0    0    0    0    0    3   -4    0    0    0    0    0    1
 1232           0.00           0.12    0    0    0    0    0    0    2    1    0    0    0    0    0    1
 1233          -0.12           0.00    0    0    0    0    0    0    2    0    0    0    0    0    0    0
 1234           0.12           0.00    0    0    0    0    0    0    1   -1    0    0    0    0    0   -2
 1235           0.12           0.00    0    0    0    0    0    0    1   -4    0    0    0    0    0   -1
 1236          -0.12           0.00    0    0    0    0    0    0    0    9  -17    0    0    0    0   -2
 1237           0.00           0.12    0    0    0    0    0    0    0    7   -7    0    0    0    0    2
 1238           0.00           0.12    0    0    0    0    0    0    0    7  -12    0    0    0    0   -2
 1239          -0.12           0.00    0    0    0    0    0    0    0    6   -4    0    0    0    0    2
 1240           0.00          -0.12    0    0    0    0    0    0    0    6   -8    1    5    0    0    2
 1241          -0.12           0.00    0    0    0    0    0    0    0    6   -9    0    0    0    0   -2
 1242           0.00           0.12    0    0    0    0    0    0    0    6  -10    0    0    0    0    0
 1243          -0.12           0.00    0    0    0    0    0    0    0    5    0   -4    0    0    0    2
 1244           0.12           0.00    0    0    0    0    0    0    0    5   -6    0    0    0    0    0
 1245          -0.12           0.00    0    0    0    0    0    0    0    5   -7    0    0    0    0   -2
 1246          -0.12           0.00    0    0    0    0    0    0    0    5   -8    3    0    0    0    2
 1247          -0.12           0.00    0    0    0    0    0    0    0    5   -9    0    0    0    0   -1
 1248          -0.12           0.00    0    0    0    0    0    0    0    5  -13    0    0    0    0   -2
 1249           0.12           0.00    0    0    0    0    0    0    0    5  -16    4    5    0    0   -2
 1250          -0.12           0.00    0    0    0    0    0    0    0    4   -7    0    0    0    0   -1
 1251           0.12           0.00    0    0    0    0    0    0    0    4   -8    3    0    0    0    1
 1252           0.12           0.00    0    0    0    0    0    0    0    4   -8    3    0    0    0   -1
 1253          -0.12           0.00    0    0    0    0    0    0    0    3    0   -5    0    0    0   -2
 1254           0.00          -0.12    0    0    0    0    0    0    0    3   -5    0    0    0    0   -1
 1255           0.00          -0.12    0    0    0    0    0    0    0    3   -7    0    0    0    0   -2
 1256           0.00          -0.12    0    0    0    0    0    0    0    3   -9    0    0    0    0   -2
 1257           0.12           0.00    0    0    0    0    0    0    0    2    1    0    0    0    0    2
 1258           0.12           0.00    0    0    0    0    0    0    0    2    0    2    0    0    0    2
 1259           0.00           0.12    0    0    0    0    0    0    0    2    0    0   -3    0    0    0
 1260           0.00          -0.12    0    0    0    0    0    0    0    2   -8    1    5    0    0   -2
 1261           0.00           0.12    0    0    0    0    0    0    0    1    0    1    0    0    0    1
 1262          -0.12           0.00    0    0    0    0    0    0    0    1    0    1   -5    0    0    0
 1263           0.00          -0.12    0    0    0    0    0    0    0    1    0    0    2    0    0    2
 1264          -0.12           0.00    0    0    0    0    0    0    0    1    0    0   -3    0    0    0
 1265           0.12           0.00    0    0    0    0    0    0    0    1    0   -3    5    0    0    0
 1266           0.00           0.12    0    0    0    0    0    0    0    1   -3    0    0    0    0    0
 1267           0.12           0.00    0    0    0    0    0    0    0    0    0    2   -6    3    0   -2
 1268           0.00           0.12    0    0    0    0    0    0    0    0    0    1   -2    0    0    0
 1269           0.00           0.12    0    0    0    0    0    0    0    0    0    0    0    1    0    0
 1270           0.12           0.00    0    0    0    0    0    0    0    0    0    0    0    0    0    2
 1271          -0.12           0.00    1    1    0   -6    0    0    0    0    0    0    0    0    0    0
 1272          -0.12           0.00    1    0    0    3    0    0    0    0    0    0    0    0    0    0
 1273           0.12           0.00    2    0    0   -2    0    0    0   -2    0    0    5    0    0    0
 1274          -0.12           0.00    3    0   -2   -1   -1    0    0    0    0    0    0    0    0    0
 1275          -0.12           0.00    1    2    2   -4    1    0    0    0    0    0    0    0    0    0
 1276          -0.12           0.00    0    0    1   -1    0    0   -2    2    0    0    0    0    0    0
 1277           0.00           0.12    1    0   -1    1    0    0    0    1    0    0    0    0    0    0
 1278           0.12           0.00    0    0    2   -2    1    0    1   -1    0    0    0    0    0    0
 1279          -0.12           0.00    2    0    2   -6    1    0    0    0    0    0    0    0    0    0
 1280          -0.12           0.00    2    1    2   -2    0    0    0    0    0    0    0    0    0    0
 1281          -0.12           0.00    2    1    2    0    0    0    0    0    0    0    0    0    0    0
 1282          -0.12           0.00    1    0    2    1    0    0    0    0    0    0    0    0    0    0
 1283           0.12           0.00    4    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1284          -0.12           0.00    0    0    0    0    1    0    0    7  -13    0    0    0    0    0
 1285           0.12           0.00    2    1    0    2    1    0    0    0    0    0    0    0    0    0
 1286          -0.12           0.00    1   -1    0    4    1    0    0    0    0    0    0    0    0    0
 1287          -0.12           0.00    3    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1288           0.12           0.00    2    0    0   -6   -1    0    0    0    0    0    0    0    0    0
 1289           0.12           0.00    0    0    1   -1   -1    0   -5    7    0    0    0    0    0    0
 1290           0.12           0.00    0    0    0    0    2    0   -3    5    0    0    0    0    0    0
 1291           0.00          -0.12    0    0    1    1    2    0    0    1    0    0    0    0    0    0
 1292           0.00          -0.12    0    0    1   -1    2    0   -3    4    0    0    0    0    0    0
 1293           0.00           0.12    0    0    0    0    1    0   -1    2    0    0    0    0    0    0
 1294           0.12           0.00    2   -1    0   -2    2    0    0    0    0    0    0    0    0    0
 1295           0.12           0.00    1    1   -2    1   -2    0    0    0    0    0    0    0    0    0
 1296           0.12           0.00    2    1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1297          -0.12           0.00    0    0    1   -1    2    0    0   -1    0    0   -1    0    0    0
 1298          -0.12           0.00    2    1    2   -4    2    0    0    0    0    0    0    0    0    0
 1299           0.12           0.00    2    0    0   -4    2    0    0    0    0    0    0    0    0    0
 1300          -0.12           0.00    3    0    0    0   -2    0    0    0    0    0    0    0    0    0
 1301           0.12           0.00    1    0    0   -4    2    0    0    0    0    0    0    0    0    0
 1302           0.12           0.00    2    0    0   -4   -2    0    0    0    0    0    0    0    0    0
 1303           0.12           0.00    1    1    0    0   -2    0    0    0    0    0    0    0    0    0
 1304          -0.12           0.00    0    0    0    0    1    0    0    2    0   -2    0    0    0    0
 1305          -0.12           0.00    1    0   -2    4   -2    0    0    0    0    0    0    0    0    0
 1306           0.11           0.00    0    0    4   -4    4    0    0    0    0    0    0    0    0    0 ];
 
%j = 1  Nb of terms = 253
a1=[...
 1307       -3328.48      205833.15    0    0    0    0    1    0    0    0    0    0    0    0    0    0
 1308         197.53       12814.01    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1309          41.19        2187.91    0    0    2    0    2    0    0    0    0    0    0    0    0    0
 1310         -35.85       -2004.36    0    0    0    0    2    0    0    0    0    0    0    0    0    0
 1311          59.15         501.82    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1312          -5.82         448.76    0    0    2    0    1    0    0    0    0    0    0    0    0    0
 1313        -179.56         164.33    0    1    0    0    0    0    0    0    0    0    0    0    0    0
 1314           5.67         288.49    1    0    2    0    2    0    0    0    0    0    0    0    0    0
 1315          23.85        -214.50    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
 1316           2.87        -154.91    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1317           2.14        -119.21    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
 1318           1.17         -74.33    1    0    0    0    1    0    0    0    0    0    0    0    0    0
 1319           1.47          70.31    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1320          -0.42          58.94    1    0    2    0    1    0    0    0    0    0    0    0    0    0
 1321          -0.95          57.12    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1322          -1.08         -54.19    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1323           0.92          36.78    0    0    2    2    2    0    0    0    0    0    0    0    0    0
 1324           0.68         -31.01    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
 1325           0.74          29.60    2    0    2    0    2    0    0    0    0    0    0    0    0    0
 1326          -0.60         -27.59    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1327         -11.11         -15.07    1    0    0    0    0    0    0    0    0    0    0    0    0    0
 1328          -0.40         -24.05    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1329          -0.81          19.06    0    1    0    0    1    0    0    0    0    0    0    0    0    0
 1330           3.18          15.32    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
 1331          -0.08         -17.90    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
 1332          -0.16          15.55    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1333          -0.77          14.40    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
 1334          -0.25          11.67    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1335           6.18           3.58    0    0    1   -1    1    0    0   -1    0   -2    5    0    0    0
 1336          -1.00          -7.27    0    1    2    0    2    0    0    0    0    0    0    0    0    0
 1337          -0.99           6.87    0    1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1338          -0.27           7.49    0    0    2    2    1    0    0    0    0    0    0    0    0    0
 1339          -0.30           7.31    0    0    0    2    1    0    0    0    0    0    0    0    0    0
 1340           0.20           7.30    1    0    2    2    2    0    0    0    0    0    0    0    0    0
 1341           0.33           6.80    2    0    0   -2   -1    0    0    0    0    0    0    0    0    0
 1342           0.27          -6.81    1    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1343           0.35           6.08    0    0    0    2   -1    0    0    0    0    0    0    0    0    0
 1344           0.35           6.09    0    1   -2    2   -1    0    0    0    0    0    0    0    0    0
 1345          -0.14          -6.19    2    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1346           0.14           6.02    2    0    2    0    1    0    0    0    0    0    0    0    0    0
 1347           2.69          -2.76    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
 1348          -0.08          -4.93    2    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1349           2.85          -1.77    0    0    0    0    0    0    0    0    0    2   -5    0    0   -1
 1350          -0.07          -4.27    0    1    2   -2    1    0    0    0    0    0    0    0    0    0
 1351          -3.71           0.38    0    2    0    0    0    0    0    0    0    0    0    0    0    0
 1352           3.75           0.04    0    1   -1    1   -1    0    0    0    0    0    0    0    0    0
 1353          -0.82          -2.73    0    0    0    2    0    0    0    0    0    0    0    0    0    0
 1354          -0.06           2.93    2    0   -2    0   -2    0    0    0    0    0    0    0    0    0
 1355          -0.04           2.83    2    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1356           0.08           2.75    3    0    2    0    2    0    0    0    0    0    0    0    0    0
 1357           0.07           2.75    1   -1    2    0    2    0    0    0    0    0    0    0    0    0
 1358          -0.07           2.70    1    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1359          -0.07           2.52    0    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1360          -0.05          -2.53    2    0    0    0    1    0    0    0    0    0    0    0    0    0
 1361          -0.04           2.40    1    0   -2    2   -1    0    0    0    0    0    0    0    0    0
 1362          -0.06          -2.37    1    1    2    0    2    0    0    0    0    0    0    0    0    0
 1363           0.69          -1.45    0    0    1   -1    1    0   -8   12    0    0    0    0    0    0
 1364          -0.04           2.00    0    0    0    0    0    0    3   -5    0    0    0    0    0   -2
 1365           1.99           0.02    1    0   -1    0   -1    0    0    0    0    0    0    0    0    0
 1366          -0.94           1.07    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
 1367           0.04           1.91    1    0    0    0    2    0    0    0    0    0    0    0    0    0
 1368          -0.58          -1.36    2    0    0    0    0    0    0    0    0    0    0    0    0    0
 1369          -0.51          -1.25    0    0    2    0    0    0    0    0    0    0    0    0    0    0
 1370          -0.04          -1.59    0    0    2    1    2    0    0    0    0    0    0    0    0    0
 1371           0.39          -1.23    1    0    0   -1    0    0    0    0    0    0    0    0    0    0
 1372           0.03          -1.57    1   -1    0   -1   -1    0    0    0    0    0    0    0    0    0
 1373          -0.03           1.50    0    2   -2    2   -1    0    0    0    0    0    0    0    0    0
 1374           0.04           1.48    1    0    2    2    1    0    0    0    0    0    0    0    0    0
 1375          -0.04           1.45    1    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1376           0.02          -1.36    1    0    0    0   -2    0    0    0    0    0    0    0    0    0
 1377           0.03          -1.32    2    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1378          -0.03          -1.24    1    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1379          -0.02          -1.18    2    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1380          -0.03           1.16    2    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1381           0.02           1.13    0    0    0    0    0    0    0    0    0    2    0    0    0    2
 1382           0.04          -1.11    1    0    0   -1   -1    0    0    0    0    0    0    0    0    0
 1383           0.02           1.11    1    0    0    2    1    0    0    0    0    0    0    0    0    0
 1384           0.03          -1.10    1    0   -4    0   -2    0    0    0    0    0    0    0    0    0
 1385           0.03           1.04    2    0    2    2    2    0    0    0    0    0    0    0    0    0
 1386          -0.51           0.56    0    0    0    0    1    0    0   -1    2    0    0    0    0    0
 1387           0.02          -0.98    0    0    2   -2   -1    0    0    0    0    0    0    0    0    0
 1388          -0.02          -0.94    0    1    2    0    1    0    0    0    0    0    0    0    0    0
 1389          -0.02          -0.89    3    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1390          -0.02          -0.88    0    0    4   -2    2    0    0    0    0    0    0    0    0    0
 1391           0.31           0.60    0    0    0    0    0    0    8  -13    0    0    0    0    0   -1
 1392           0.02          -0.87    1    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1393          -0.02          -0.87    2    0   -2    0    1    0    0    0    0    0    0    0    0    0
 1394          -0.01           0.83    0    0    2    0   -1    0    0    0    0    0    0    0    0    0
 1395          -0.02           0.77    0    1   -2    0   -1    0    0    0    0    0    0    0    0    0
 1396           0.42          -0.36    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
 1397          -0.01          -0.73    0    1    0    0    2    0    0    0    0    0    0    0    0    0
 1398           0.01           0.71    1    1    0   -2    1    0    0    0    0    0    0    0    0    0
 1399           0.01           0.68    0    0    2   -1    2    0    0    0    0    0    0    0    0    0
 1400           0.02           0.66    0    0    2    4    2    0    0    0    0    0    0    0    0    0
 1401          -0.01          -0.62    1   -1    0    0    1    0    0    0    0    0    0    0    0    0
 1402          -0.01           0.62    1    0    0    2   -1    0    0    0    0    0    0    0    0    0
 1403          -0.58          -0.03    0    0    1   -1    1    0    0   -1    0    2   -5    0    0    0
 1404          -0.01           0.58    1   -1    0   -1   -2    0    0    0    0    0    0    0    0    0
 1405           0.44           0.14    0    0    0    0    0    0    0    2   -8    3    0    0    0   -2
 1406           0.02           0.56    1   -1    2    2    2    0    0    0    0    0    0    0    0    0
 1407           0.01          -0.57    0    1    0    0   -2    0    0    0    0    0    0    0    0    0
 1408          -0.13          -0.45    1    0    0    2    0    0    0    0    0    0    0    0    0    0
 1409           0.01           0.56    3    0    2    0    1    0    0    0    0    0    0    0    0    0
 1410           0.01          -0.55    1   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1411           0.01           0.55    1    0    2   -4    1    0    0    0    0    0    0    0    0    0
 1412          -0.52           0.03    0    0    2   -2    1    0   -5    6    0    0    0    0    0    0
 1413          -0.01           0.54    1    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1414          -0.01          -0.51    0    1    2    2    2    0    0    0    0    0    0    0    0    0
 1415          -0.41          -0.11    0    0    0    0    0    0    0    6   -8    3    0    0    0    2
 1416          -0.01           0.50    0    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1417           0.01           0.48    1   -1    2    0    1    0    0    0    0    0    0    0    0    0
 1418           0.45          -0.04    1    0   -2   -2    0    0    0    0    0    0    0    0    0    0
 1419           0.01          -0.48    0    0    0    0    0    0    4   -6    0    0    0    0    0   -2
 1420           0.01           0.46    2   -1    2    0    2    0    0    0    0    0    0    0    0    0
 1421          -0.23           0.24    2    0   -2    0    0    0    0    0    0    0    0    0    0    0
 1422           0.01           0.46    0    0    0    1    1    0    0    0    0    0    0    0    0    0
 1423           0.35          -0.11    0    0    0    0    1    0    0   -4    8   -3    0    0    0    0
 1424           0.01           0.45    0    0    0    2    2    0    0    0    0    0    0    0    0    0
 1425           0.01          -0.45    1   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1426           0.00          -0.45    0    0    0    0    0    0    0    3    0   -1    0    0    0    2
 1427          -0.01           0.44    0    0    0    0    0    0    2   -4    0    0    0    0    0   -2
 1428           0.35           0.09    0    0    0    0    1    0    0    4   -8    3    0    0    0    0
 1429           0.01           0.42    1    1    0    0    1    0    0    0    0    0    0    0    0    0
 1430          -0.01          -0.41    1    1    2    0    1    0    0    0    0    0    0    0    0    0
 1431           0.09          -0.33    0    0    1   -1    1    0    0    0   -2    0    0    0    0    0
 1432           0.00           0.41    0    1    0   -2    1    0    0    0    0    0    0    0    0    0
 1433           0.01           0.40    0    3    2   -2    2    0    0    0    0    0    0    0    0    0
 1434          -0.01          -0.39    2    1    2    0    2    0    0    0    0    0    0    0    0    0
 1435          -0.39          -0.01    1    0   -1    0   -2    0    0    0    0    0    0    0    0    0
 1436           0.01          -0.39    0    0    0    1   -1    0    0    0    0    0    0    0    0    0
 1437          -0.01           0.38    1    0    0   -2   -2    0    0    0    0    0    0    0    0    0
 1438           0.32          -0.07    0    0    0    0    0    0    0    0    0    2    0    0    0    1
 1439          -0.01           0.36    1   -1    0    0   -1    0    0    0    0    0    0    0    0    0
 1440          -0.01          -0.36    0    0    0    0    0    0    2    0    0    0    0    0    0    2
 1441           0.01          -0.34    1    0   -2    2   -2    0    0    0    0    0    0    0    0    0
 1442           0.01          -0.34    2    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
 1443           0.01           0.33    0    0    0    0    0    0    1    1    0    0    0    0    0    2
 1444          -0.01          -0.32    1    0    2    1    2    0    0    0    0    0    0    0    0    0
 1445           0.01           0.32    1    0    2   -1    2    0    0    0    0    0    0    0    0    0
 1446          -0.01          -0.32    0    0    2    1    1    0    0    0    0    0    0    0    0    0
 1447          -0.01          -0.31    1    0   -2    0    1    0    0    0    0    0    0    0    0    0
 1448          -0.31           0.00    0    0    1    0    1    0    0    0    0    0    0    0    0    0
 1449          -0.07          -0.24    1    0    2    0    0    0    0    0    0    0    0    0    0    0
 1450           0.10          -0.21    0    0    1   -1    1    0    0   -1    0    0   -1    0    0    0
 1451          -0.01          -0.30    1    0    0   -2    2    0    0    0    0    0    0    0    0    0
 1452          -0.01           0.29    1    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
 1453          -0.01          -0.29    1    1    2   -2    1    0    0    0    0    0    0    0    0    0
 1454           0.00           0.29    1    0   -2    1   -1    0    0    0    0    0    0    0    0    0
 1455           0.23           0.06    0    0    1   -1    1    0    0   -1    0   -1    0    0    0    0
 1456           0.26           0.02    0    0    0    0    0    0    0    0    0    2   -5    0    0    1
 1457           0.00          -0.27    2    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1458           0.25           0.02    0    0    0    0    0    0    5   -8    0    0    0    0    0   -2
 1459           0.09          -0.18    0    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1460           0.01           0.25    0    0    0    0    0    0    0    1    0    1    0    0    0    2
 1461           0.14          -0.11    1    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1462           0.00          -0.25    1    1    0    0   -1    0    0    0    0    0    0    0    0    0
 1463           0.01           0.24    4    0    2    0    2    0    0    0    0    0    0    0    0    0
 1464          -0.01          -0.24    2    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1465           0.00           0.23    2    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
 1466           0.01           0.23    0    1    2    1    2    0    0    0    0    0    0    0    0    0
 1467          -0.01          -0.23    1    0    4   -2    2    0    0    0    0    0    0    0    0    0
 1468           0.00          -0.23    0    1    0    2    1    0    0    0    0    0    0    0    0    0
 1469           0.00          -0.22    1    0   -4    0   -1    0    0    0    0    0    0    0    0    0
 1470           0.00           0.21    1    0    0   -4   -1    0    0    0    0    0    0    0    0    0
 1471           0.01           0.21    2    0    2    2    1    0    0    0    0    0    0    0    0    0
 1472          -0.17           0.03    0    0    1   -1    1    0    0   -1    0    2    0    0    0    0
 1473          -0.17           0.03    0    0    0    0    0    0    8  -13    0    0    0    0    0   -2
 1474           0.00          -0.19    0    0    0    0    0    0    5   -7    0    0    0    0    0   -2
 1475           0.14          -0.06    0    0    0    0    0    0    0    0    0    0    2    0    0    1
 1476           0.03          -0.17    0    0    1   -1    1    0   -5    7    0    0    0    0    0    0
 1477          -0.13           0.06    0    0    1   -1    1    0    0   -1    0    0    2    0    0    0
 1478           0.00           0.19    0    0    2   -4    1    0    0    0    0    0    0    0    0    0
 1479           0.00           0.19    0    0    2   -3    2    0    0    0    0    0    0    0    0    0
 1480          -0.06          -0.13    1    0    2    0    2    0    0    1    0    0    0    0    0    0
 1481           0.00           0.18    0    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1482          -0.09          -0.09    1   -1    0    0    0    0    0    0    0    0    0    0    0    0
 1483           0.10          -0.09    0    0    0    0    0    0    0    8  -16    4    5    0    0   -2
 1484           0.06           0.12    1    0   -2    0   -2    0    0    4   -8    3    0    0    0    0
 1485           0.00           0.18    0    0    0    0    0    0    0    4    0   -2    0    0    0    2
 1486           0.00          -0.18    0    0    4    0    2    0    0    0    0    0    0    0    0    0
 1487           0.00           0.17    0    0    0    4    1    0    0    0    0    0    0    0    0    0
 1488          -0.03           0.15    0    0    0    0    0    0    0    2    0   -1    0    0    0    2
 1489          -0.01          -0.16    0    0    0    0    0    0    0    2    0    1    0    0    0    2
 1490           0.00           0.17    1    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1491           0.00          -0.17    2    0    0   -2   -2    0    0    0    0    0    0    0    0    0
 1492           0.11           0.06    0    0    0    0    0    0    8  -11    0    0    0    0    0   -2
 1493           0.00          -0.17    3    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1494          -0.08           0.09    1    0   -2    0    0    0    0    0    0    0    0    0    0    0
 1495          -0.17           0.00    0    0    0    0    0    0    2   -1    0    0    0    0    0    2
 1496           0.00          -0.16    0    0    4   -2    1    0    0    0    0    0    0    0    0    0
 1497           0.01           0.15    1    0    2    4    2    0    0    0    0    0    0    0    0    0
 1498          -0.13          -0.03    0    0    0    0    0    0    1   -1    0    0    0    0    0   -1
 1499           0.00           0.15    1    1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1500           0.00           0.15    2    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1501          -0.13           0.03    0    0    0    0    0    0    4   -6    0    0    0    0    0   -1
 1502           0.10          -0.06    0    0    0    0    0    0    8  -15    0    0    0    0    0   -2
 1503          -0.07           0.08    0    0    0    0    0    0    0    1    2    0    0    0    0    2
 1504          -0.09          -0.06    0    0    0    0    0    0    0    3    0   -2    0    0    0    2
 1505           0.00           0.15    3    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1506          -0.07          -0.08    0    0    0    0    0    0    0    3   -2    0    0    0    0    2
 1507           0.00          -0.14    2    0    0   -4   -1    0    0    0    0    0    0    0    0    0
 1508           0.02           0.12    0    0    0    0    0    0    0    0    0    3    0    0    0    2
 1509           0.07           0.08    1    1    0    0    0    0    0    0    0    0    0    0    0    0
 1510          -0.03          -0.11    3    0    0    0    0    0    0    0    0    0    0    0    0    0
 1511          -0.01          -0.14    0    0    0    0    0    0    0    4   -2    0    0    0    0    2
 1512           0.00          -0.14    0    0    2   -2    1    0   -3    3    0    0    0    0    0    0
 1513           0.02          -0.12    1    0    0   -1    1    0    0    0    0    0    0    0    0    0
 1514           0.00          -0.14    1    1    2    2    2    0    0    0    0    0    0    0    0    0
 1515           0.00           0.14    0    0    1   -1    1    0    0    3   -8    3    0    0    0    0
 1516           0.00           0.14    0    0    2   -1    1    0    0    0    0    0    0    0    0    0
 1517           0.00           0.13    0    0    2    4    1    0    0    0    0    0    0    0    0    0
 1518           0.08          -0.06    0    0    0    0    1    0    8  -13    0    0    0    0    0    0
 1519           0.00           0.13    1   -1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1520           0.00           0.13    0    0    0    0    0    0    0    0    0    0    2    0    0    2
 1521           0.01           0.13    3    0    2    2    2    0    0    0    0    0    0    0    0    0
 1522           0.00           0.13    2    0    0    2    1    0    0    0    0    0    0    0    0    0
 1523           0.00           0.13    1   -1   -2    2   -1    0    0    0    0    0    0    0    0    0
 1524          -0.02          -0.11    2    0    0   -2    1    0    0   -2    0    3    0    0    0    0
 1525           0.08          -0.04    0    0    0    1    0    0    0    0    0    0    0    0    0    0
 1526           0.00           0.13    2    0    0   -4    1    0    0    0    0    0    0    0    0    0
 1527           0.00           0.13    1    0    0   -4    1    0    0    0    0    0    0    0    0    0
 1528           0.01          -0.12    2    1   -2    0   -1    0    0    0    0    0    0    0    0    0
 1529           0.00           0.12    0    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1530          -0.02          -0.11    0    0    0    0    0    0    5   -8    0    0    0    0    0   -1
 1531           0.00          -0.12    2    0    2   -4    1    0    0    0    0    0    0    0    0    0
 1532           0.00          -0.12    0    0    0    0    0    0    0    8  -16    4    5    0    0    2
 1533           0.00          -0.12    0    0    1   -1    1    0    0   -5    8   -3    0    0    0    0
 1534           0.04           0.08    1    0    0    0   -1    0  -18   16    0    0    0    0    0    0
 1535           0.00          -0.12    0    0    0    0    0    0    0    1    0   -3    0    0    0   -2
 1536           0.00           0.12    1   -1    0   -2   -2    0    0    0    0    0    0    0    0    0
 1537           0.00          -0.12    0    0    0    2   -2    0    0    0    0    0    0    0    0    0
 1538           0.00          -0.11    0    0    0    0    0    0    6   -8    0    0    0    0    0   -2
 1539           0.03          -0.09    1    0    0   -4    0    0    0    0    0    0    0    0    0    0
 1540           0.00           0.11    0    2    0    0    1    0    0    0    0    0    0    0    0    0
 1541          -0.11           0.00    0    0    0    0    0    0    3   -2    0    0    0    0    0    2
 1542           0.00           0.11    1   -1    2    2    1    0    0    0    0    0    0    0    0    0
 1543           0.00          -0.11    1   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
 1544           0.07           0.05    0    0    0    0    1    0   -8   13    0    0    0    0    0    0
 1545           0.11           0.00    0    0    0    0    0    0    2   -5    0    0    0    0    0   -2
 1546           0.00          -0.11    0    0    0    0    0    0    1   -3    0    0    0    0    0   -2
 1547           0.00          -0.11    4    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1548           0.02          -0.09    0    0    0    0    1    0    0    0    0   -2    5    0    0    0
 1549           0.00           0.11    2    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1550           0.02           0.09    0    0    0    0    1    0    0    0    0    2   -5    0    0    0
 1551           0.00          -0.11    0    2    2   -2    1    0    0    0    0    0    0    0    0    0
 1552           0.00           0.11    1    0    2    0   -1    0    0    0    0    0    0    0    0    0
 1553          -0.08          -0.02    0    0    0    0    1    0    0    0    0    1    0    0    0    0
 1554           0.00          -0.10    2    1    0   -2    1    0    0    0    0    0    0    0    0    0
 1555           0.00          -0.10    2   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
 1556          -0.03          -0.07    1    0    0    0    1    0  -18   16    0    0    0    0    0    0
 1557           0.00           0.10    1    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1558           0.00           0.10    2    0    0   -2    2    0    0    0    0    0    0    0    0    0
 1559           0.00          -0.10    1   -1   -2   -2   -1    0    0    0    0    0    0    0    0    0];

%j = 2  Nb of terms = 36
a2=[...
 1560        2038.00          82.26    0    0    0    0    1    0    0    0    0    0    0    0    0    0
 1561         155.75          -2.70    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1562          26.92          -0.45    0    0    2    0    2    0    0    0    0    0    0    0    0    0
 1563         -24.43           0.46    0    0    0    0    2    0    0    0    0    0    0    0    0    0
 1564         -17.36          -0.50    0    1    0    0    0    0    0    0    0    0    0    0    0    0
 1565          -8.41           0.01    1    0    0    0    0    0    0    0    0    0    0    0    0    0
 1566           6.08          -1.36    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1567           4.59           0.17    0    0    2    0    1    0    0    0    0    0    0    0    0    0
 1568           3.57          -0.06    1    0    2    0    2    0    0    0    0    0    0    0    0    0
 1569           2.54           0.60    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
 1570           1.86           0.00    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
 1571          -1.52          -0.07    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1572           1.46           0.04    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
 1573          -0.75          -0.02    1    0    0    0    1    0    0    0    0    0    0    0    0    0
 1574          -0.75           0.00    0    0    0    2    0    0    0    0    0    0    0    0    0    0
 1575          -0.71          -0.01    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1576          -0.69           0.02    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1577           0.61           0.02    1    0    2    0    1    0    0    0    0    0    0    0    0    0
 1578           0.54          -0.04    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1579          -0.56           0.00    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
 1580           0.46          -0.02    0    0    2    2    2    0    0    0    0    0    0    0    0    0
 1581           0.38          -0.01    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
 1582           0.37          -0.02    2    0    2    0    2    0    0    0    0    0    0    0    0    0
 1583          -0.34           0.01    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1584          -0.35           0.00    2    0    0    0    0    0    0    0    0    0    0    0    0    0
 1585          -0.31           0.00    0    0    2    0    0    0    0    0    0    0    0    0    0    0
 1586           0.19          -0.09    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
 1587           0.26           0.00    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
 1588           0.24          -0.01    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1589          -0.20           0.00    0    2    0    0    0    0    0    0    0    0    0    0    0    0
 1590           0.18          -0.01    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
 1591           0.17           0.00    0    1    0    0    1    0    0    0    0    0    0    0    0    0
 1592           0.15           0.01    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1593          -0.15           0.00    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
 1594          -0.13           0.00    2    0   -2    0    0    0    0    0    0    0    0    0    0    0
 1595          -0.12           0.00    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0];
 
%j = 3  Nb of terms = 4
a3=[...
 1596           1.76         -20.39    0    0    0    0    1    0    0    0    0    0    0    0    0    0
 1597           0.00          -1.27    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1598           0.00          -0.22    0    0    2    0    2    0    0    0    0    0    0    0    0    0
 1599           0.00           0.20    0    0    0    0    2    0    0    0    0    0    0    0    0    0];

%j = 4  Nb of terms = 1
a4=[...      
 1600          -0.10          -0.02    0    0    0    0    1    0    0    0    0    0    0    0    0    0];

% *  ----------------------------------
% *  The series for Y: numerical values
% *  ----------------------------------
% 
% *  Polynomial coefficients
      Ypol = [     -0.00695078,...
                   -0.02538199,...
                  -22.40725099,...
                    0.00184228,...
                    0.00111306,...
                    0.00000099];
                
% Expression for the Y coordinate of the CIP in the GCRS based on the IAU2000A 
% precession-nutation model 
% 
% --------------------------------------------------------------------------------------------------------------
% 
% Y = polynomial part + non-polynomial part
% 
% Non-polynomial part (unit microarcsecond)
% (ARG being for various combination of the fundamental arguments of the nutation theory)
% 
%   Sum_i[b_{c,0})_i * cos(ARG) + b_{s,0})_i * sin(ARG)] 
% 
% + Sum_i)j=1,4 [b_{c,j})_i * t^j * cos(ARG) + b_{s,j})_i * sin(ARG)] * t^j]
% 
% The Table below provides the values for b_{s,j})_i and b_{c,j})_i
% 
% The expressions for the fundamental arguments appearing in columns 4 to 8 (luni-solar part) 
% and in columns 6 to 17 (planetary part) are those of the IERS Conventions 2000
% 
% --------------------------------------------------------------------------------------------------------------
% 
%     i    b_{s,j})_i      b_{c,j})_i    l    l'   F    D   Om L_Me L_Ve  L_E L_Ma  L_J L_Sa  L_U L_Ne  p_A
% 
% --------------------------------------------------------------------------------------------------------------

%j = 0  Nb of terms = 962
b0=[...
    1        1538.18     9205236.26    0    0    0    0    1    0    0    0    0    0    0    0    0    0
    2        -458.66      573033.42    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
    3         137.41       97846.69    0    0    2    0    2    0    0    0    0    0    0    0    0    0
    4         -29.05      -89618.24    0    0    0    0    2    0    0    0    0    0    0    0    0    0
    5         -17.40       22438.42    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
    6          31.80       20069.50    0    0    2    0    1    0    0    0    0    0    0    0    0    0
    7          36.70       12902.66    1    0    2    0    2    0    0    0    0    0    0    0    0    0
    8         -13.20       -9592.72    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
    9        -192.40        7387.02    0    1    0    0    0    0    0    0    0    0    0    0    0    0
   10           3.92       -6918.22    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
   11           0.40       -5331.13    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
   12          -0.90       -3323.89    1    0    0    0    1    0    0    0    0    0    0    0    0    0
   13           7.50        3143.98    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
   14           7.80        2636.13    1    0    2    0    1    0    0    0    0    0    0    0    0    0
   15          -6.60        2554.51    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
   16          -2.00       -2423.59    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   17           6.80        1645.01    0    0    2    2    2    0    0    0    0    0    0    0    0    0
   18           0.00       -1387.00    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
   19           5.90        1323.81    2    0    2    0    2    0    0    0    0    0    0    0    0    0
   20          -0.30       -1233.89    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
   21           0.30       -1075.60    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   22          -4.48         852.85    0    1    0    0    1    0    0    0    0    0    0    0    0    0
   23           0.10        -800.34    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
   24          35.80        -674.99    1    0    0    0    0    0    0    0    0    0    0    0    0    0
   25          -1.40         695.54    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
   26          -0.50         684.99    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
   27          -2.62         643.75    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
   28          -1.50         522.11    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
   29         273.50         164.70    0    0    1   -1    1    0    0   -1    0   -2    5    0    0    0
   30           1.40         335.24    0    0    2    2    1    0    0    0    0    0    0    0    0    0
   31           1.90         326.60    1    0    2    2    2    0    0    0    0    0    0    0    0    0
   32           0.40         327.11    0    0    0    2    1    0    0    0    0    0    0    0    0    0
   33          -0.50        -325.03    0    1    2    0    2    0    0    0    0    0    0    0    0    0
   34          -0.40         307.03    0    1   -2    0   -2    0    0    0    0    0    0    0    0    0
   35           0.50         304.17    2    0    0   -2   -1    0    0    0    0    0    0    0    0    0
   36          -0.10        -304.46    1    0    2   -2    1    0    0    0    0    0    0    0    0    0
   37          -0.40        -276.81    2    0    2   -2    2    0    0    0    0    0    0    0    0    0
   38           0.90         272.05    0    0    0    2   -1    0    0    0    0    0    0    0    0    0
   39           0.30         272.22    0    1   -2    2   -1    0    0    0    0    0    0    0    0    0
   40           1.20         269.45    2    0    2    0    1    0    0    0    0    0    0    0    0    0
   41           0.10        -220.67    2    0    0   -2    1    0    0    0    0    0    0    0    0    0
   42         128.60         -77.10    0    0    0    0    0    0    0    0    0    2   -5    0    0   -1
   43           0.10        -190.79    0    1    2   -2    1    0    0    0    0    0    0    0    0    0
   44         167.90           0.00    0    1   -1    1   -1    0    0    0    0    0    0    0    0    0
   45          -8.20        -123.48    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
   46           0.10         131.04    2    0   -2    0   -2    0    0    0    0    0    0    0    0    0
   47           0.40         126.64    2    0    0    0   -1    0    0    0    0    0    0    0    0    0
   48           2.90        -122.28    0    0    0    2    0    0    0    0    0    0    0    0    0    0
   49           0.70         123.20    3    0    2    0    2    0    0    0    0    0    0    0    0    0
   50           0.40         123.20    1   -1    2    0    2    0    0    0    0    0    0    0    0    0
   51          -0.30         120.70    1    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
   52          -0.50         112.90    0    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
   53          -0.20        -112.94    2    0    0    0    1    0    0    0    0    0    0    0    0    0
   54           0.20         107.31    1    0   -2    2   -1    0    0    0    0    0    0    0    0    0
   55          -0.30        -106.20    1    1    2    0    2    0    0    0    0    0    0    0    0    0
   56          31.90         -64.10    0    0    1   -1    1    0   -8   12    0    0    0    0    0    0
   57           0.00          89.50    0    0    0    0    0    0    3   -5    0    0    0    0    0   -2
   58          89.10           0.00    1    0   -1    0   -1    0    0    0    0    0    0    0    0    0
   59           0.00          85.32    1    0    0    0    2    0    0    0    0    0    0    0    0    0
   60          -0.20         -71.00    0    0    2    1    2    0    0    0    0    0    0    0    0    0
   61           0.00         -70.01    1   -1    0   -1   -1    0    0    0    0    0    0    0    0    0
   62          13.90         -55.30    1    0    0   -1    0    0    0    0    0    0    0    0    0    0
   63           0.00          67.25    0    2   -2    2   -1    0    0    0    0    0    0    0    0    0
   64           0.40          66.29    1    0    2    2    1    0    0    0    0    0    0    0    0    0
   65          -0.40          64.70    1    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
   66           1.30         -60.90    2    0    0    0    0    0    0    0    0    0    0    0    0    0
   67          -0.20         -60.92    1    0    0    0   -2    0    0    0    0    0    0    0    0    0
   68           0.20         -59.20    2    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
   69           1.10         -55.55    0    0    2    0    0    0    0    0    0    0    0    0    0    0
   70           0.00         -55.60    1    1    2   -2    2    0    0    0    0    0    0    0    0    0
   71          -0.10         -52.69    2    0    2   -2    1    0    0    0    0    0    0    0    0    0
   72          -0.20          51.80    2    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
   73           1.00         -49.51    1    0    0   -1   -1    0    0    0    0    0    0    0    0    0
   74           0.00          50.50    0    0    0    0    0    0    0    0    0    2    0    0    0    2
   75           2.50          47.70    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
   76           0.10          49.59    1    0    0    2    1    0    0    0    0    0    0    0    0    0
   77           0.10         -49.00    1    0   -4    0   -2    0    0    0    0    0    0    0    0    0
   78         -23.20          24.60    0    0    0    0    1    0    0   -1    2    0    0    0    0    0
   79           0.40          46.50    2    0    2    2    2    0    0    0    0    0    0    0    0    0
   80          -0.10         -44.04    0    0    2   -2   -1    0    0    0    0    0    0    0    0    0
   81          -0.10         -42.19    0    1    2    0    1    0    0    0    0    0    0    0    0    0
   82          13.30          26.90    0    0    0    0    0    0    8  -13    0    0    0    0    0   -1
   83          -0.10         -39.90    3    0    2   -2    2    0    0    0    0    0    0    0    0    0
   84          -0.10         -39.50    0    0    4   -2    2    0    0    0    0    0    0    0    0    0
   85           0.00         -39.11    1    1    0   -2   -1    0    0    0    0    0    0    0    0    0
   86          -0.10         -38.92    2    0   -2    0    1    0    0    0    0    0    0    0    0    0
   87           0.10          36.95    0    0    2    0   -1    0    0    0    0    0    0    0    0    0
   88          -0.10          34.59    0    1   -2    0   -1    0    0    0    0    0    0    0    0    0
   89           0.20         -32.55    0    1    0    0    2    0    0    0    0    0    0    0    0    0
   90          -0.10          31.61    1    1    0   -2    1    0    0    0    0    0    0    0    0    0
   91           0.00          30.40    0    0    2   -1    2    0    0    0    0    0    0    0    0    0
   92           0.20          29.40    0    0    2    4    2    0    0    0    0    0    0    0    0    0
   93           0.00         -27.91    1   -1    0    0    1    0    0    0    0    0    0    0    0    0
   94           0.10          27.50    1    0    0    2   -1    0    0    0    0    0    0    0    0    0
   95         -25.70          -1.70    0    0    1   -1    1    0    0   -1    0    2   -5    0    0    0
   96          19.90           5.90    0    0    0    0    0    0    0    2   -8    3    0    0    0   -2
   97           0.00          25.80    1   -1    0   -1   -2    0    0    0    0    0    0    0    0    0
   98           0.20          25.20    1   -1    2    2    2    0    0    0    0    0    0    0    0    0
   99           0.00         -25.31    0    1    0    0   -2    0    0    0    0    0    0    0    0    0
  100           0.20          25.00    3    0    2    0    1    0    0    0    0    0    0    0    0    0
  101          -0.10          24.40    1    0    2   -4    1    0    0    0    0    0    0    0    0    0
  102           0.10         -24.40    1   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  103         -23.30           0.90    0    0    2   -2    1    0   -5    6    0    0    0    0    0    0
  104          -0.10          24.00    1    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  105         -18.00          -5.30    0    0    0    0    0    0    0    6   -8    3    0    0    0    2
  106          -0.10         -22.80    0    1    2    2    2    0    0    0    0    0    0    0    0    0
  107          -0.10          22.50    0    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  108           0.10          21.60    1   -1    2    0    1    0    0    0    0    0    0    0    0    0
  109           0.00         -21.30    0    0    0    0    0    0    4   -6    0    0    0    0    0   -2
  110           0.10          20.70    2   -1    2    0    2    0    0    0    0    0    0    0    0    0
  111           0.70         -20.10    0    0    0    0    0    0    0    3    0   -1    0    0    0    2
  112           0.00          20.51    0    0    0    1    1    0    0    0    0    0    0    0    0    0
  113          15.90          -4.50    0    0    0    0    1    0    0   -4    8   -3    0    0    0    0
  114           0.20         -19.94    1    0    0    2    0    0    0    0    0    0    0    0    0    0
  115           0.00          20.11    0    0    0    2    2    0    0    0    0    0    0    0    0    0
  116          15.60           4.40    0    0    0    0    1    0    0    4   -8    3    0    0    0    0
  117           0.00         -20.00    1   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
  118           0.00          19.80    0    0    0    0    0    0    2   -4    0    0    0    0    0   -2
  119           0.00          18.91    1    1    0    0    1    0    0    0    0    0    0    0    0    0
  120           4.30         -14.60    0    0    1   -1    1    0    0    0   -2    0    0    0    0    0
  121          -0.10         -18.50    1    1    2    0    1    0    0    0    0    0    0    0    0    0
  122          -0.10          18.40    0    1    0   -2    1    0    0    0    0    0    0    0    0    0
  123           0.00          18.10    0    3    2   -2    2    0    0    0    0    0    0    0    0    0
  124           1.00          16.81    0    2    0    0    0    0    0    0    0    0    0    0    0    0
  125          -0.10         -17.60    2    1    2    0    2    0    0    0    0    0    0    0    0    0
  126         -17.60           0.00    1    0   -1    0   -2    0    0    0    0    0    0    0    0    0
  127          -1.30         -16.26    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
  128           0.00         -17.41    0    0    0    1   -1    0    0    0    0    0    0    0    0    0
  129          14.50          -2.70    0    0    0    0    0    0    0    0    0    2    0    0    0    1
  130           0.00          17.08    1    0    0   -2   -2    0    0    0    0    0    0    0    0    0
  131           0.00          16.21    1   -1    0    0   -1    0    0    0    0    0    0    0    0    0
  132           0.00         -16.00    0    0    0    0    0    0    2    0    0    0    0    0    0    2
  133           0.00         -15.31    1    0   -2    2   -2    0    0    0    0    0    0    0    0    0
  134           0.00         -15.10    2    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
  135           0.00          14.70    0    0    0    0    0    0    1    1    0    0    0    0    0    2
  136           0.00          14.40    1    0    2   -1    2    0    0    0    0    0    0    0    0    0
  137          -0.10         -14.30    1    0    2    1    2    0    0    0    0    0    0    0    0    0
  138           0.00         -14.40    0    0    2    1    1    0    0    0    0    0    0    0    0    0
  139           0.00         -13.81    1    0   -2    0    1    0    0    0    0    0    0    0    0    0
  140           4.50          -9.30    0    0    1   -1    1    0    0   -1    0    0   -1    0    0    0
  141         -13.80           0.00    0    0    1    0    1    0    0    0    0    0    0    0    0    0
  142           0.00         -13.38    1    0    0   -2    2    0    0    0    0    0    0    0    0    0
  143          -0.10          13.10    1    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
  144          10.30           2.70    0    0    1   -1    1    0    0   -1    0   -1    0    0    0    0
  145           0.00          12.80    1    0   -2    1   -1    0    0    0    0    0    0    0    0    0
  146           0.00         -12.80    1    1    2   -2    1    0    0    0    0    0    0    0    0    0
  147          11.70           0.80    0    0    0    0    0    0    0    0    0    2   -5    0    0    1
  148           0.00         -12.00    2    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  149          11.30           0.50    0    0    0    0    0    0    5   -8    0    0    0    0    0   -2
  150           0.00          11.40    0    0    0    0    0    0    0    1    0    1    0    0    0    2
  151           0.00         -11.20    1    1    0    0   -1    0    0    0    0    0    0    0    0    0
  152           0.10          10.90    4    0    2    0    2    0    0    0    0    0    0    0    0    0
  153           0.10         -10.77    1    0    2    0    0    0    0    0    0    0    0    0    0    0
  154           0.00         -10.80    2    1    2   -2    2    0    0    0    0    0    0    0    0    0
  155          -0.20          10.47    2    0   -2    0    0    0    0    0    0    0    0    0    0    0
  156           0.00          10.50    2    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
  157           0.00         -10.40    1    0    4   -2    2    0    0    0    0    0    0    0    0    0
  158           0.00          10.40    0    1    2    1    2    0    0    0    0    0    0    0    0    0
  159           0.00         -10.20    0    1    0    2    1    0    0    0    0    0    0    0    0    0
  160           0.00         -10.00    1    0   -4    0   -1    0    0    0    0    0    0    0    0    0
  161           0.00           9.60    1    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  162           0.10           9.40    2    0    2    2    1    0    0    0    0    0    0    0    0    0
  163          -7.60           1.70    0    0    0    0    0    0    8  -13    0    0    0    0    0   -2
  164          -7.70           1.40    0    0    1   -1    1    0    0   -1    0    2    0    0    0    0
  165           1.40          -7.50    0    0    1   -1    1    0   -5    7    0    0    0    0    0    0
  166           6.10          -2.70    0    0    0    0    0    0    0    0    0    0    2    0    0    1
  167           0.00          -8.70    0    0    0    0    0    0    5   -7    0    0    0    0    0   -2
  168          -5.90           2.60    0    0    1   -1    1    0    0   -1    0    0    2    0    0    0
  169           0.00           8.40    0    0    2   -4    1    0    0    0    0    0    0    0    0    0
  170          -0.20          -8.11    0    1    0   -2    0    0    0    0    0    0    0    0    0    0
  171          -2.60          -5.70    1    0    2    0    2    0    0    1    0    0    0    0    0    0
  172           0.00           8.30    0    0    2   -3    2    0    0    0    0    0    0    0    0    0
  173           2.70           5.50    1    0   -2    0   -2    0    0    4   -8    3    0    0    0    0
  174           4.20          -4.00    0    0    0    0    0    0    0    8  -16    4    5    0    0   -2
  175          -0.10           8.00    0    0    0    0    0    0    0    4    0   -2    0    0    0    2
  176           0.00           8.09    0    1    0   -2   -1    0    0    0    0    0    0    0    0    0
  177          -1.30           6.70    0    0    0    0    0    0    0    2    0   -1    0    0    0    2
  178           0.00          -7.90    0    0    4    0    2    0    0    0    0    0    0    0    0    0
  179           0.00           7.80    0    0    0    4    1    0    0    0    0    0    0    0    0    0
  180          -7.50          -0.20    0    0    0    0    0    0    2   -1    0    0    0    0    0    2
  181          -0.50          -7.20    0    0    0    0    0    0    0    2    0    1    0    0    0    2
  182           4.90           2.70    0    0    0    0    0    0    8  -11    0    0    0    0    0   -2
  183           0.00           7.50    1    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  184           0.00          -7.50    3    0    2   -2    1    0    0    0    0    0    0    0    0    0
  185           0.00          -7.49    2    0    0   -2   -2    0    0    0    0    0    0    0    0    0
  186           0.00          -7.20    0    0    4   -2    1    0    0    0    0    0    0    0    0    0
  187           0.10           6.90    1    0    2    4    2    0    0    0    0    0    0    0    0    0
  188          -5.60           1.40    0    0    0    0    0    0    4   -6    0    0    0    0    0   -1
  189          -5.70          -1.30    0    0    0    0    0    0    1   -1    0    0    0    0    0   -1
  190           0.00           6.90    2    1    0   -2   -1    0    0    0    0    0    0    0    0    0
  191           4.20          -2.70    0    0    0    0    0    0    8  -15    0    0    0    0    0   -2
  192           0.00           6.90    1    1   -2    0   -2    0    0    0    0    0    0    0    0    0
  193          -3.10           3.70    0    0    0    0    0    0    0    1    2    0    0    0    0    2
  194          -3.90          -2.90    0    0    0    0    0    0    0    3    0   -2    0    0    0    2
  195           0.00           6.60    3    0    0    0   -1    0    0    0    0    0    0    0    0    0
  196          -3.10          -3.50    0    0    0    0    0    0    0    3   -2    0    0    0    0    2
  197           1.10          -5.39    1    0    0   -1    1    0    0    0    0    0    0    0    0    0
  198           0.00          -6.40    2    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  199           0.90           5.50    0    0    0    0    0    0    0    0    0    3    0    0    0    2
  200           0.00          -6.30    0    0    2   -2    1    0   -3    3    0    0    0    0    0    0
  201          -0.10          -6.20    0    0    0    0    0    0    0    4   -2    0    0    0    0    2
  202           0.00          -6.10    1    1    2    2    2    0    0    0    0    0    0    0    0    0
  203           0.00           6.10    0    0    1   -1    1    0    0    3   -8    3    0    0    0    0
  204           0.00           6.10    0    0    2   -1    1    0    0    0    0    0    0    0    0    0
  205           3.50          -2.50    0    0    0    0    1    0    8  -13    0    0    0    0    0    0
  206           0.00           6.00    0    0    2    4    1    0    0    0    0    0    0    0    0    0
  207           0.00           5.90    1   -1    0   -2   -1    0    0    0    0    0    0    0    0    0
  208          -0.90          -4.80    2    0    0   -2    1    0    0   -2    0    3    0    0    0    0
  209           0.00           5.70    1   -1   -2    2   -1    0    0    0    0    0    0    0    0    0
  210           0.10           5.60    3    0    2    2    2    0    0    0    0    0    0    0    0    0
  211           0.00           5.70    0    0    0    0    0    0    0    0    0    0    2    0    0    2
  212           0.00           5.70    2    0    0    2    1    0    0    0    0    0    0    0    0    0
  213           0.00           5.60    2    0    0   -4    1    0    0    0    0    0    0    0    0    0
  214           0.00           5.60    1    0    0   -4    1    0    0    0    0    0    0    0    0    0
  215           0.20          -5.40    2    1   -2    0   -1    0    0    0    0    0    0    0    0    0
  216          -0.90          -4.70    0    0    0    0    0    0    5   -8    0    0    0    0    0   -1
  217          -0.40          -5.10    1    1    0   -2    0    0    0    0    0    0    0    0    0    0
  218           0.00           5.50    0    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  219           0.00          -5.40    2    0    2   -4    1    0    0    0    0    0    0    0    0    0
  220           0.00          -5.40    0    0    0    0    0    0    0    8  -16    4    5    0    0    2
  221           1.80           3.60    1    0    0    0   -1    0  -18   16    0    0    0    0    0    0
  222           0.00           5.30    1   -1    0   -2   -2    0    0    0    0    0    0    0    0    0
  223           0.00          -5.30    0    0    1   -1    1    0    0   -5    8   -3    0    0    0    0
  224           0.00          -5.20    0    0    0    0    0    0    0    1    0   -3    0    0    0   -2
  225           0.00          -5.19    0    0    0    2   -2    0    0    0    0    0    0    0    0    0
  226           3.00           2.10    0    0    0    0    1    0   -8   13    0    0    0    0    0    0
  227           0.00          -5.10    0    0    0    0    0    0    6   -8    0    0    0    0    0   -2
  228           0.00           5.07    0    2    0    0    1    0    0    0    0    0    0    0    0    0
  229           0.90          -4.10    0    0    0    0    1    0    0    0    0   -2    5    0    0    0
  230          -5.00           0.00    0    0    0    0    0    0    3   -2    0    0    0    0    0    2
  231           0.00          -5.00    3    0    0    0    0    0    0    0    0    0    0    0    0    0
  232           0.00           5.00    1   -1    2    2    1    0    0    0    0    0    0    0    0    0
  233           0.00          -5.00    1   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
  234           0.00          -4.90    4    0    2   -2    2    0    0    0    0    0    0    0    0    0
  235           4.90           0.00    0    0    0    0    0    0    2   -5    0    0    0    0    0   -2
  236           0.00          -4.90    0    0    0    0    0    0    1   -3    0    0    0    0    0   -2
  237           0.90           3.90    0    0    0    0    1    0    0    0    0    2   -5    0    0    0
  238           0.00           4.80    2    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  239          -3.70          -1.10    0    0    0    0    1    0    0    0    0    1    0    0    0    0
  240           0.00          -4.72    0    2    2   -2    1    0    0    0    0    0    0    0    0    0
  241           0.00           4.71    1    0    2    0   -1    0    0    0    0    0    0    0    0    0
  242           0.00          -4.50    2    1    0   -2    1    0    0    0    0    0    0    0    0    0
  243          -1.50          -3.00    1    0    0    0    1    0  -18   16    0    0    0    0    0    0
  244           0.00          -4.50    2   -1   -2    0   -1    0    0    0    0    0    0    0    0    0
  245           0.30          -4.11    1   -1    0    0    0    0    0    0    0    0    0    0    0    0
  246           0.00           4.40    1    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  247           0.00          -4.40    1   -1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  248           0.00           4.39    2    0    0   -2    2    0    0    0    0    0    0    0    0    0
  249           0.00          -4.30    3    0    0    0    1    0    0    0    0    0    0    0    0    0
  250           0.00           4.30    2    0   -2    2   -1    0    0    0    0    0    0    0    0    0
  251           0.00          -4.30    0    1    2    2    1    0    0    0    0    0    0    0    0    0
  252           0.20           4.03    1    0   -2    0    0    0    0    0    0    0    0    0    0    0
  253           0.20           4.00    0    2   -2    0   -2    0    0    0    0    0    0    0    0    0
  254          -0.60           3.50    0    0    1   -1    1    0    0   -1    0    0    1    0    0    0
  255           0.00           4.10    3    0   -2    0   -1    0    0    0    0    0    0    0    0    0
  256           0.00           4.00    1    0   -4    2   -2    0    0    0    0    0    0    0    0    0
  257           0.00          -4.00    0    1   -2    1   -2    0    0    0    0    0    0    0    0    0
  258           0.00          -3.91    1    0    0   -4    0    0    0    0    0    0    0    0    0    0
  259           1.90           2.00    0    0    0    0    1    0    0    1   -2    0    0    0    0    0
  260           0.00           3.90    2   -1    2    2    2    0    0    0    0    0    0    0    0    0
  261           0.00           3.90    0    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  262           0.00          -3.90    1    0   -2   -3   -2    0    0    0    0    0    0    0    0    0
  263           3.10          -0.80    0    0    0    0    0    0    2   -4    0    0    0    0    0   -1
  264           0.00           3.90    2   -1    2    0    1    0    0    0    0    0    0    0    0    0
  265           0.00           3.90    2    0    0    0    2    0    0    0    0    0    0    0    0    0
  266           0.00           3.80    0    0    0    0    0    0    0    2    0    0    0    0    0    2
  267          -0.20           3.51    1    1    0    0    0    0    0    0    0    0    0    0    0    0
  268           0.00          -3.60    1    0   -2   -1   -2    0    0    0    0    0    0    0    0    0
  269          -2.10           1.50    0    0    0    0    0    0    0    1    0    2    0    0    0    2
  270           0.00          -3.60    1    1   -2    1   -1    0    0    0    0    0    0    0    0    0
  271           0.70           2.80    0    0    0    0    0    0    3   -7    0    0    0    0    0   -2
  272          -2.80           0.70    0    0    0    0    0    0    0    0    0    1    0    0    0   -1
  273           0.00          -3.50    0    0    0    0    0    0    0    2    0    0    0    0    0    0
  274          -2.90          -0.60    0    0    0    0    0    0    2   -2    0    0    0    0    0   -1
  275           0.00          -3.40    1   -1    0   -2    1    0    0    0    0    0    0    0    0    0
  276           0.00           3.40    2    0   -4    0   -2    0    0    0    0    0    0    0    0    0
  277           0.00           3.36    0    0    2    0    3    0    0    0    0    0    0    0    0    0
  278           0.50           2.80    2    0    0   -2   -1    0    0   -2    0    3    0    0    0    0
  279           2.60          -0.70    0    0    0    0    0    0    3   -5    0    0    0    0    0   -1
  280           1.00          -2.30    0    0    0    0    0    0    0    0    0    1    0    0    0    2
  281           0.00          -3.30    2    1    2    0    1    0    0    0    0    0    0    0    0    0
  282           0.00           3.30    1    1   -2    0   -1    0    0    0    0    0    0    0    0    0
  283           0.00           3.23    0    2    0    0   -1    0    0    0    0    0    0    0    0    0
  284           0.00           3.20    1   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  285           0.00          -3.20    0    2    2    0    2    0    0    0    0    0    0    0    0    0
  286           0.00          -3.20    0    0    0    0    0    0    7   -9    0    0    0    0    0   -2
  287           0.00           3.20    0    0    0    0    0    0    0    0    2    0    0    0    0    2
  288           2.90          -0.30    0    0    0    0    0    0    0    0    0    1    0    0    0    1
  289           0.08           3.05    0    0    2   -2    3    0    0    0    0    0    0    0    0    0
  290          -0.70          -2.40    0    0    0    0    0    0    3   -3    0    0    0    0    0    2
  291           0.00          -3.08    1    0    2   -4    2    0    0    0    0    0    0    0    0    0
  292           0.00           3.00    2    0    2   -1    2    0    0    0    0    0    0    0    0    0
  293          -1.60           1.40    1    0    0   -1    1    0    0   -1    0    2    0    0    0    0
  294          -2.90          -0.10    0    0    0    0    0    0    4   -7    0    0    0    0    0   -2
  295           0.00          -2.90    2    0    0   -2    1    0    0   -2    0    2    0    0    0    0
  296          -2.50           0.40    0    0    1   -1    1    0    0   -2    2    0    0    0    0    0
  297           0.40          -2.50    0    0    0    0    0    0    0    0    0    0    1    0    0    1
  298           0.00          -2.90    1    0    2    1    1    0    0    0    0    0    0    0    0    0
  299           0.00           2.89    0    1   -2    2    0    0    0    0    0    0    0    0    0    0
  300           0.00          -2.80    2    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  301          -2.50           0.30    0    0    1   -1    1    0    0   -1    0    1    0    0    0    0
  302          -2.50          -0.30    0    0    0    0    0    0    0    1   -2    0    0    0    0   -1
  303           0.00          -2.70    0    0    2   -2    1    0    0   -2    0    2    0    0    0    0
  304           2.70           0.00    1    0   -1    0   -3    0    0    0    0    0    0    0    0    0
  305           0.00          -2.60    0    1    0    1    1    0    0    0    0    0    0    0    0    0
  306           0.00          -2.60    0    1    0    2   -1    0    0    0    0    0    0    0    0    0
  307           0.00           2.60    3   -1    2    0    2    0    0    0    0    0    0    0    0    0
  308           2.10           0.50    0    0    1   -1    1    0   -3    4    0    0    0    0    0    0
  309           0.00           2.50    0    1    0    1   -1    0    0    0    0    0    0    0    0    0
  310           0.80           1.70    0    0    0    0    0    0    0    0    0    0    1    0    0   -1
  311           1.90          -0.60    0    0    0    0    0    0    0    6  -16    4    5    0    0   -2
  312           0.00          -2.50    2    0    0    0   -2    0    0    0    0    0    0    0    0    0
  313           0.00          -2.40    2    0    0    2    0    0    0    0    0    0    0    0    0    0
  314           0.00           2.40    2    0    0    2   -1    0    0    0    0    0    0    0    0    0
  315           0.00          -2.40    1    1   -2    2   -1    0    0    0    0    0    0    0    0    0
  316           0.00           2.40    1    0    2   -3    1    0    0    0    0    0    0    0    0    0
  317          -1.90           0.50    0    0    0    0    0    0    5   -7    0    0    0    0    0   -1
  318          -0.10          -2.30    2    0    0   -4    0    0    0    0    0    0    0    0    0    0
  319           0.00           2.30    1    1    2   -4    1    0    0    0    0    0    0    0    0    0
  320           0.00          -2.30    3    1    2    0    2    0    0    0    0    0    0    0    0    0
  321          -1.40           0.90    0    0    0    0    0    0    0    0    4    0    0    0    0    2
  322          -0.10          -2.20    0    0    0    0    0    0    8  -10    0    0    0    0    0   -2
  323          -0.20          -2.00    0    0    0    1    0    0    0    0    0    0    0    0    0    0
  324           0.00           2.20    1   -2    2    0    2    0    0    0    0    0    0    0    0    0
  325           0.00          -2.20    2   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  326           0.00           2.20    1    1    2    1    2    0    0    0    0    0    0    0    0    0
  327           0.00           2.20    0    0    0    0    0    0    0    2   -4    0    0    0    0   -2
  328          -1.80          -0.40    0    0    0    0    0    0    3   -3    0    0    0    0    0   -1
  329           0.00           2.20    0    1    2    1    1    0    0    0    0    0    0    0    0    0
  330           0.00           2.20    4    0    2    0    1    0    0    0    0    0    0    0    0    0
  331          -1.70           0.40    0    0    0    0    0    0    1   -1    0    0    0    0    0    1
  332          -0.80          -1.30    0    0    0    0    0    0    0    5   -4    0    0    0    0    2
  333          -1.30          -0.80    0    0    0    0    0    0    0    4   -4    0    0    0    0    2
  334           0.00           2.10    1    0   -2    1   -2    0    0    0    0    0    0    0    0    0
  335           0.00           2.10    1   -1    2   -2    1    0    0    0    0    0    0    0    0    0
  336           0.00          -2.10    1    0    4   -2    1    0    0    0    0    0    0    0    0    0
  337           0.00          -2.10    2    1    2   -2    1    0    0    0    0    0    0    0    0    0
  338           0.00           2.10    1    0    2   -1    1    0    0    0    0    0    0    0    0    0
  339           0.00          -2.00    1    0    2   -2   -1    0    0    0    0    0    0    0    0    0
  340           0.00           2.00    1    0   -2    2    1    0    0    0    0    0    0    0    0    0
  341           0.00           2.00    1   -1    0    2   -1    0    0    0    0    0    0    0    0    0
  342           0.00           2.00    2   -2    0   -2   -1    0    0    0    0    0    0    0    0    0
  343           0.00          -2.00    1    2    2   -2    2    0    0    0    0    0    0    0    0    0
  344           2.00           0.00    0    0    0    0    0    0    6   -9    0    0    0    0    0   -2
  345           1.10          -0.90    0    0    0    0    0    0    0    0    0    2   -5    0    0   -2
  346           1.60          -0.40    0    0    0    0    0    0    1   -3    0    0    0    0    0   -1
  347           0.00          -1.91    1    0   -2   -2    0    0    0    0    0    0    0    0    0    0
  348           0.00          -1.90    0    0    0    4    0    0    0    0    0    0    0    0    0    0
  349           0.00           1.90    0    0    4   -4    2    0    0    0    0    0    0    0    0    0
  350           0.00          -1.90    3    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
  351           0.00           1.90    2   -1    0   -2    1    0    0    0    0    0    0    0    0    0
  352           1.50           0.40    0    0    1   -1    1    0   -4    5    0    0    0    0    0    0
  353          -1.50          -0.40    0    0    0    0    0    0    0    2    0    0    0    0    0    1
  354          -1.40          -0.50    0    0    1   -1    1    0    0   -1    0   -4   10    0    0    0
  355          -1.00           0.90    0    0    1   -1    1    0    0   -1    0    0    0    0    2    0
  356           0.00          -1.90    2    0    2    1    2    0    0    0    0    0    0    0    0    0
  357          -0.30           1.60    0    0    0    0    0    0    0    4    0   -3    0    0    0    2
  358           0.00           1.90    0    0    0    4   -1    0    0    0    0    0    0    0    0    0
  359           0.00           1.90    1   -1    0    2    1    0    0    0    0    0    0    0    0    0
  360           0.00          -1.80    2   -1    0    0    1    0    0    0    0    0    0    0    0    0
  361           0.00          -1.80    2    0   -4   -2   -2    0    0    0    0    0    0    0    0    0
  362          -1.10           0.70    0    0    0    0    0    0    0    2   -4    0    0    0    0   -1
  363           0.20          -1.60    0    0    0    0    0    0    2   -3    0    0    0    0    0   -1
  364           0.00           1.80    2    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  365           0.00          -1.71    0    0    2    2    0    0    0    0    0    0    0    0    0    0
  366          -1.20          -0.50    0    0    2   -2    1    0    0   -9   13    0    0    0    0    0
  367           1.50           0.20    0    0    1    1    1    0    0    1    0    0    0    0    0    0
  368          -0.60          -1.10    1    0    2    0    1    0    0   -2    0    3    0    0    0    0
  369           0.60           1.10    1    0   -2    0   -1    0    0   -1    0    0    0    0    0    0
  370          -0.60          -1.10    0    0    0    0    0    0    0    4   -3    0    0    0    0    2
  371          -1.10           0.60    0    0    0    0    1    0    0   -2    4    0    0    0    0    0
  372          -1.70           0.00    0    0    0    0    1    0    2   -3    0    0    0    0    0    0
  373           0.00           1.60    0    1   -2    2    1    0    0    0    0    0    0    0    0    0
  374           0.00          -1.60    2    0    4   -2    2    0    0    0    0    0    0    0    0    0
  375           0.00          -1.60    0    0    2    3    2    0    0    0    0    0    0    0    0    0
  376           1.20          -0.40    0    0    0    0    0    0    0    0    0    3    0    0    0    1
  377          -0.50          -1.10    2    0    2    0    2    0    0    2    0   -3    0    0    0    0
  378           0.60           1.00    0    0    0    0    0    0    0    3   -6    0    0    0    0   -2
  379          -1.30          -0.30    0    0    0    0    0    0    4   -4    0    0    0    0    0   -1
  380           0.30          -1.30    0    0    0    0    0    0    0    1   -2    0    0    0    0    1
  381           0.00           1.60    1    0    0    4    1    0    0    0    0    0    0    0    0    0
  382           0.00          -1.60    1    1    0    2    1    0    0    0    0    0    0    0    0    0
  383           0.00          -1.60    0    0    4    0    1    0    0    0    0    0    0    0    0    0
  384           1.10          -0.50    0    0    1   -1    2    0    0   -1    0    0    2    0    0    0
  385           0.00          -1.50    1   -1    0    2    0    0    0    0    0    0    0    0    0    0
  386           0.00          -1.50    0    1    2   -2   -1    0    0    0    0    0    0    0    0    0
  387           0.00           1.50    2   -1    0    0   -1    0    0    0    0    0    0    0    0    0
  388           0.00          -1.50    1    0    0    1   -1    0    0    0    0    0    0    0    0    0
  389           0.00          -1.50    1    2    0   -2   -1    0    0    0    0    0    0    0    0    0
  390           1.50           0.00    1    0    0   -1    1    0   -3    4    0    0    0    0    0    0
  391           0.00          -1.50    0    1    4   -2    2    0    0    0    0    0    0    0    0    0
  392           1.30          -0.20    0    0    0    0    1    0    0    0    0    0    1    0    0    0
  393           0.00          -1.50    0    0    0    0    0    0    9  -11    0    0    0    0    0   -2
  394          -1.20          -0.30    0    0    0    0    0    0    1    1    0    0    0    0    0    1
  395          -1.40           0.10    0    0    0    0    0    0    0    4    0   -1    0    0    0    2
  396          -0.50           1.00    0    0    2    0    2    0    0    1    0    0    0    0    0    0
  397          -0.50           1.00    0    0    0    0    0    0    0    1   -8    3    0    0    0   -2
  398           0.20          -1.30    0    0    0    0    0    0    0    1    0   -4    0    0    0   -2
  399           0.00           1.50    1    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  400           0.00           1.50    1    0    0   -1   -2    0    0    0    0    0    0    0    0    0
  401           0.00           1.50    0    0    0    0    1    0    1   -1    0    0    0    0    0    0
  402           0.00           1.49    2    0    2   -4    2    0    0    0    0    0    0    0    0    0
  403           0.00          -1.41    2    0    2    0    0    0    0    0    0    0    0    0    0    0
  404           0.00           1.41    0    1    0    2    0    0    0    0    0    0    0    0    0    0
  405           0.00          -1.40    0    0    2   -3    1    0    0    0    0    0    0    0    0    0
  406           0.00          -1.40    1   -2   -2   -2   -2    0    0    0    0    0    0    0    0    0
  407           0.00           1.40    1    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  408           0.00          -1.40    0    0    2   -2    1    0   -4    4    0    0    0    0    0    0
  409           1.10          -0.30    0    0    1   -1    1    0   -1    0    0    0    0    0    0    0
  410           0.00          -1.40    3    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  411           0.00           1.40    1    0    2    4    1    0    0    0    0    0    0    0    0    0
  412           1.40           0.00    0    0    1    0    2    0    0    0    0    0    0    0    0    0
  413          -0.30           1.10    0    0    0    0    1    0   -3    5    0    0    0    0    0    0
  414           0.20           1.20    2   -1    0   -2    0    0    0    0    0    0    0    0    0    0
  415          -1.30           0.00    1    0    0   -1   -1    0   -3    4    0    0    0    0    0    0
  416           0.00          -1.30    1   -1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  417           0.00           1.30    1   -2    2    2    2    0    0    0    0    0    0    0    0    0
  418          -0.70          -0.60    0    0    1   -1    1    0    8  -14    0    0    0    0    0    0
  419          -0.80           0.50    0    0    1   -1    1    0    3   -6    0    0    0    0    0    0
  420          -0.20          -1.10    0    0    0    0    0    0    6  -10    0    0    0    0    0   -2
  421           1.10           0.20    0    0    0    0    0    0    0    2    0   -2    0    0    0    1
  422           0.00          -1.30    3    1    2   -2    2    0    0    0    0    0    0    0    0    0
  423           0.00          -1.30    1    0   -4   -2   -2    0    0    0    0    0    0    0    0    0
  424           0.00          -1.30    0    0    0    0    0    0    3   -1    0    0    0    0    0    2
  425           0.00          -1.30    0    0    0    0    0    0    0    3    0    0   -1    0    0    2
  426           0.00          -1.29    0    0    2   -4    2    0    0    0    0    0    0    0    0    0
  427           0.00           1.20    1    2    0   -2    1    0    0    0    0    0    0    0    0    0
  428           0.00          -1.20    2    1    0    0   -1    0    0    0    0    0    0    0    0    0
  429          -0.40          -0.80    0    0    0    0    1    0    0    8  -15    0    0    0    0    0
  430           0.00           1.20    2    0    2    4    2    0    0    0    0    0    0    0    0    0
  431           1.20           0.00    0    0    0    0    0    0    3   -6    0    0    0    0    0   -2
  432          -0.70          -0.50    0    0    1   -1    1    0    0   -3    4    0    0    0    0    0
  433          -1.00           0.20    0    0    0    0    0    0    6   -8    0    0    0    0    0   -1
  434          -1.00           0.20    0    0    0    0    0    0    2    1    0    0    0    0    0    2
  435           0.20          -1.00    0    0    0    0    0    0    2    1    0    0    0    0    0    1
  436           0.40           0.80    0    0    0    0    0    0    0    1   -4    0    0    0    0   -2
  437          -0.40           0.80    0    0    0    0    0    0    0    0    3    0    0    0    0    2
  438           0.00          -1.20    1    1    2    2    1    0    0    0    0    0    0    0    0    0
  439           0.00           1.15    0    0    0    0    3    0    0    0    0    0    0    0    0    0
  440           0.00           1.10    2    1    0    0    1    0    0    0    0    0    0    0    0    0
  441          -0.20           0.90    2    0    0   -2    1    0   -6    8    0    0    0    0    0    0
  442          -1.10           0.00    0    0    3    0    3    0    0    0    0    0    0    0    0    0
  443           0.00          -1.10    2    1    2    2    2    0    0    0    0    0    0    0    0    0
  444          -1.10           0.00    1    0    1    0    1    0    0    0    0    0    0    0    0    0
  445           0.00           1.10    2   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  446           0.00           1.10    3    0    2   -4    2    0    0    0    0    0    0    0    0    0
  447           0.00           1.10    0    0    0    0    0    0    0    7   -8    3    0    0    0    2
  448           0.60          -0.50    0    0    0    0    0    0    0    0    0    0    0    0    2    1
  449          -0.90          -0.20    0    0    0    0    0    0    5   -5    0    0    0    0    0   -1
  450          -0.40          -0.70    0    0    0    0    0    0    0    4   -7    0    0    0    0   -2
  451          -0.50           0.60    1   -2    2   -2    1    0    0    0    0    0    0    0    0    0
  452           0.00           1.10    3    0    2    2    1    0    0    0    0    0    0    0    0    0
  453           0.00          -1.10    1   -1   -2    2   -2    0    0    0    0    0    0    0    0    0
  454           0.00           1.00    0    0    0    0    1    0   -1    1    0    0    0    0    0    0
  455           1.00           0.00    0    0    0    0    1    0   -2    3    0    0    0    0    0    0
  456           0.80          -0.20    0    0    0    0    1    0    0    0    0   -1    0    0    0    0
  457           0.00           1.00    1    0    0    1    1    0    0    0    0    0    0    0    0    0
  458           0.00           1.00    0    2    0   -2    1    0    0    0    0    0    0    0    0    0
  459           0.00          -1.00    3    0    0   -4   -1    0    0    0    0    0    0    0    0    0
  460          -1.00           0.00    1    0    1   -2    1    0    0    0    0    0    0    0    0    0
  461           0.00           1.00    1   -1    2    4    2    0    0    0    0    0    0    0    0    0
  462           1.00           0.00    0    0    1   -1    1    0    0   -9   15    0    0    0    0    0
  463           1.00           0.00    0    0    0    0    0    0    7  -10    0    0    0    0    0   -2
  464          -0.80          -0.20    0    0    0    0    0    0    2    0    0    0    0    0    0    1
  465           0.40           0.60    0    0    0    0    0    0    3   -5    4    0    0    0    0    2
  466          -0.40          -0.60    0    0    0    0    0    0    3   -9    4    0    0    0    0   -2
  467           0.00          -1.00    1    0    4    0    2    0    0    0    0    0    0    0    0    0
  468           0.00           1.00    0    1    2   -4    1    0    0    0    0    0    0    0    0    0
  469           0.00           1.00    1    0   -4    2   -1    0    0    0    0    0    0    0    0    0
  470           0.00           1.00    0    2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  471           0.00           1.00    2    0   -2    0    2    0    0    0    0    0    0    0    0    0
  472           0.00          -1.00    1    0    0    2    2    0    0    0    0    0    0    0    0    0
  473           0.00           0.91    1    0    2   -2    0    0    0    0    0    0    0    0    0    0
  474           0.10           0.80    1   -1    0   -2    0    0    0    0    0    0    0    0    0    0
  475           0.00           0.90    1    0   -2   -2    1    0    0    0    0    0    0    0    0    0
  476           0.00           0.90    0    1    0   -2   -2    0    0    0    0    0    0    0    0    0
  477           0.00          -0.90    2    0    0   -2    1    0   -3    3    0    0    0    0    0    0
  478           0.00          -0.90    1    2    2    0    2    0    0    0    0    0    0    0    0    0
  479          -0.70          -0.20    0    0    0    0    0    0    6   -6    0    0    0    0    0   -1
  480           0.70          -0.20    0    0    0    0    0    0    1    2    0    0    0    0    0    2
  481          -0.30           0.60    0    0    0    0    0    0    0    8  -15    0    0    0    0   -2
  482           0.00           0.90    5    0    2    0    2    0    0    0    0    0    0    0    0    0
  483           0.00           0.90    3    0   -2   -6   -2    0    0    0    0    0    0    0    0    0
  484           0.00          -0.90    0    0    1   -1    1    0   -2    2    0    0    0    0    0    0
  485          -0.50          -0.40    0    0    0    0    0    0    0    3    0   -3    0    0    0    2
  486          -0.90           0.00    0    0    0    0    0    0    0    2    0    0   -1    0    0    2
  487           0.00          -0.90    1   -1    2   -1    2    0    0    0    0    0    0    0    0    0
  488           0.00           0.90    1    0   -2    4   -1    0    0    0    0    0    0    0    0    0
  489           0.00           0.90    2    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  490           0.00          -0.90    4    0    2   -2    1    0    0    0    0    0    0    0    0    0
  491           0.00          -0.90    1   -1    0   -1    0    0    0    0    0    0    0    0    0    0
  492           0.00          -0.80    2   -1    0    0    0    0    0    0    0    0    0    0    0    0
  493           0.00           0.80    3    0    0   -2   -1    0    0    0    0    0    0    0    0    0
  494           0.00          -0.80    2   -1   -2   -4   -2    0    0    0    0    0    0    0    0    0
  495           0.10           0.70    0    0    0    0    0    0    7  -11    0    0    0    0    0   -2
  496          -0.70           0.10    0    0    0    0    0    0    2   -2    0    0    0    0    0    1
  497          -0.60           0.20    0    0    0    0    0    0    7   -9    0    0    0    0    0   -1
  498           0.20           0.60    0    0    0    0    0    0    4   -7    0    0    0    0    0   -1
  499           0.00           0.80    0    0    4   -4    1    0    0    0    0    0    0    0    0    0
  500          -0.50           0.30    0    0    0    0    0    0    3   -5    0    0    0    0    0    1
  501          -0.50          -0.30    0    0    0    0    0    0    0    5   -5    0    0    0    0    2
  502          -0.50          -0.30    0    0    0    0    0    0    0    5   -9    0    0    0    0   -2
  503           0.00          -0.80    0    0    0    0    0    0    0    3   -1    0    0    0    0    2
  504          -0.30           0.50    0    0    0    0    0    0    0    2    0    2   -5    0    0    2
  505          -0.80           0.00    0    0    0    0    0    0    0    2    0    0    1    0    0    2
  506          -0.30          -0.50    0    0    0    0    0    0    0    2    0   -2    5    0    0    2
  507          -0.30           0.50    0    0    0    0    1    0   -3    7   -4    0    0    0    0    0
  508          -0.30          -0.50    0    0    0    0    1    0    3   -7    4    0    0    0    0    0
  509           0.00           0.80    0    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  510           0.00          -0.80    3    0   -2    0   -2    0    0    0    0    0    0    0    0    0
  511           0.00          -0.80    1    0   -2   -3   -1    0    0    0    0    0    0    0    0    0
  512           0.00          -0.80    0    1   -2    1   -1    0    0    0    0    0    0    0    0    0
  513           0.00           0.80    1    2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  514           0.00           0.80    1    1    0   -2   -2    0    0    0    0    0    0    0    0    0
  515           0.00          -0.80    0    0    0    0    1    0    0    1    0   -1    0    0    0    0
  516           0.00           0.76    1    0    2    0    3    0    0    0    0    0    0    0    0    0
  517           0.00           0.70    3    0    0   -4    1    0    0    0    0    0    0    0    0    0
  518           0.10          -0.60    2    0    0   -2   -1    0   -6    8    0    0    0    0    0    0
  519           0.00           0.70    0    1   -4    2   -2    0    0    0    0    0    0    0    0    0
  520           0.70           0.00    0    0    1   -1    1    0    0   -1    0   -1    1    0    0    0
  521           0.00          -0.70    0    0    0    0    1    0    3   -5    0    2    0    0    0    0
  522           0.00          -0.70    0    1    2    4    2    0    0    0    0    0    0    0    0    0
  523           0.00           0.70    4    0    2    2    2    0    0    0    0    0    0    0    0    0
  524           0.00          -0.70    2    0   -2   -3   -2    0    0    0    0    0    0    0    0    0
  525          -0.70           0.00    0    0    1   -1    1    0    2   -4    0   -3    0    0    0    0
  526          -0.50           0.20    0    0    1   -1    1    0    0   -1    0    3    0    0    0    0
  527          -0.20          -0.50    0    0    0    0    0    0    9   -9    0    0    0    0    0   -1
  528           0.50          -0.20    0    0    0    0    0    0    2   -3    0    0    0    0    0   -2
  529           0.20           0.50    0    0    0    0    0    0    0   11    0    0    0    0    0    2
  530          -0.20          -0.50    0    0    0    0    0    0    0    6  -15    0    0    0    0   -2
  531           0.50          -0.20    0    0    0    0    0    0    0    3    0    2   -5    0    0    2
  532          -0.50           0.20    0    0    0    0    0    0    0    3    0    1    0    0    0    2
  533           0.00          -0.70    0    0    0    0    0    0    0    3   -5    0    0    0    0   -2
  534           0.00          -0.70    0    0    0    0    0    0    0    2    0   -4    0    0    0   -2
  535           0.70           0.00    0    0    0    0    0    0    0    0    0    0    3    0    0    2
  536          -0.60          -0.10    0    0    0    0    0    0    3   -1    0    0    0    0    0    1
  537           0.60          -0.10    0    0    0    0    0    0    3   -3    0    0    0    0    0    1
  538           0.40           0.30    0    0    0    0    0    0    0    4   -8    0    0    0    0   -2
  539           0.00           0.70    1    0    0    1    0    0    0    0    0    0    0    0    0    0
  540           0.70           0.00    0    0    1   -1    2    0    0   -2    2    0    0    0    0    0
  541           0.00           0.70    2   -1    2    2    1    0    0    0    0    0    0    0    0    0
  542           0.00           0.70    0    1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  543           0.00           0.70    1   -1    0    0    2    0    0    0    0    0    0    0    0    0
  544           0.00          -0.60    0    0    0    0    1    0    0   -1    0    1    0    0    0    0
  545           0.00           0.60    0    0    2    2   -1    0    0    0    0    0    0    0    0    0
  546           0.10          -0.50    0    0    1   -1   -1    0    0    0   -2    0    0    0    0    0
  547           0.00           0.60    2    1    0    0    0    0    0    0    0    0    0    0    0    0
  548           0.40           0.20    0    0    0    0    1    0    0    2   -4    0    0    0    0    0
  549           0.00           0.60    1    0    0   -4   -2    0    0    0    0    0    0    0    0    0
  550           0.00          -0.60    2    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  551           0.00           0.60    1    1    0   -4    1    0    0    0    0    0    0    0    0    0
  552           0.00          -0.60    0    1    0    1    0    0    0    0    0    0    0    0    0    0
  553           0.00          -0.60    1   -1    2   -4    1    0    0    0    0    0    0    0    0    0
  554           0.50           0.10    0    0    1   -1    1    0   -2    3    0    0    0    0    0    0
  555          -0.50          -0.10    0    0    0    0    0    0    7   -7    0    0    0    0    0   -1
  556          -0.10          -0.50    0    0    0    0    0    0    4   -4    0    0    0    0    0    2
  557           0.10           0.50    0    0    0    0    0    0    1   -2    0    0    0    0    0    1
  558           0.50          -0.10    0    0    0    0    0    0    0    5    0   -2    0    0    0    2
  559          -0.10           0.50    0    0    0    0    0    0    0    3   -6    0    0    0    0   -1
  560           0.00          -0.60    2    0    0   -2   -1    0    0   -2    0    0    5    0    0    0
  561          -0.40           0.20    2    0   -1   -1   -1    0    0   -1    0    3    0    0    0    0
  562           0.00          -0.60    1    0   -2   -2   -2    0    0   -2    0    2    0    0    0    0
  563           0.60           0.00    0    0    2   -2    2    0   -8   11    0    0    0    0    0    0
  564           0.00          -0.60    0    0    2   -2    1    0   -2    2    0    0    0    0    0    0
  565           0.20           0.40    0    0    1   -1    1    0    0   -1    0    0    3    0    0    0
  566          -0.40           0.20    0    0    0    0    0    1    0   -4    0    0    0    0    0   -2
  567           0.30           0.30    0    0    0    0    0    0    8  -13    0    0    0    0    0    1
  568           0.40          -0.20    0    0    0    0    0    0    0    6    0    0    0    0    0    2
  569          -0.40          -0.20    0    0    0    0    0    0    0    6   -6    0    0    0    0    2
  570           0.00           0.60    0    0    0    0    0    0    0    1    1    0    0    0    0    2
  571           0.00           0.60    0    0    0    0    0    0    0    1    0    0    1    0    0    2
  572           0.40           0.20    0    0    0    0    0    0    0    1   -5    0    0    0    0   -2
  573          -0.20          -0.40    0    0    0    0    0    0    0    0    0    0    3    0    0    1
  574           0.00           0.60    4    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  575           0.00          -0.60    0    0    0    3    1    0    0    0    0    0    0    0    0    0
  576           0.00           0.60    3    0    0    2    1    0    0    0    0    0    0    0    0    0
  577           0.00           0.60    1    1    0   -4   -1    0    0    0    0    0    0    0    0    0
  578           0.00          -0.60    0    0    0    1    2    0    0    0    0    0    0    0    0    0
  579           0.00          -0.60    1    1    0   -2    2    0    0    0    0    0    0    0    0    0
  580           0.00          -0.60    2    0   -2    2   -2    0    0    0    0    0    0    0    0    0
  581           0.00          -0.50    1    1    0   -4    0    0    0    0    0    0    0    0    0    0
  582           0.00           0.50    2    0    2    0   -1    0    0    0    0    0    0    0    0    0
  583           0.10           0.40    2    1    0   -2    0    0    0    0    0    0    0    0    0    0
  584           0.00          -0.50    0    1    0    2    2    0    0    0    0    0    0    0    0    0
  585           0.10           0.40    0    0    0    0    1    0    3   -5    0    0    0    0    0    0
  586           0.00           0.50    1   -2    2   -2    2    0    0    0    0    0    0    0    0    0
  587           0.00           0.50    2    1    0   -4    1    0    0    0    0    0    0    0    0    0
  588           0.00          -0.50    1   -2    0    0    1    0    0    0    0    0    0    0    0    0
  589           0.30          -0.20    2    0    0   -2    1    0    0   -6    8    0    0    0    0    0
  590          -0.20           0.30    2    0    0   -2   -1    0    0   -5    6    0    0    0    0    0
  591           0.20           0.30    2    0   -1   -1   -1    0    0    3   -7    0    0    0    0    0
  592           0.40          -0.10    0    0    2    0    2    0    0   -4    8   -3    0    0    0    0
  593           0.40           0.10    0    0    2    0    2    0    0    4   -8    3    0    0    0    0
  594           0.00          -0.50    0    1   -2   -1   -2    0    0    0    0    0    0    0    0    0
  595           0.00          -0.50    3   -1    2   -2    2    0    0    0    0    0    0    0    0    0
  596           0.30           0.20    0    0    0    0    0    0    1    0    0    0    0    0    0    2
  597          -0.30           0.20    0    0    0    0    0    0    0    7  -13    0    0    0    0   -2
  598           0.20           0.30    0    0    0    0    0    0    0    2   -5    0    0    0    0   -2
  599          -0.30           0.20    0    0    0    0    0    0    0    1    0    3    0    0    0    2
  600           0.00           0.50    3   -1    2    2    2    0    0    0    0    0    0    0    0    0
  601           0.00           0.50    3   -1   -2   -1   -2    0    0    0    0    0    0    0    0    0
  602           0.00           0.50    0    0    2    6    2    0    0    0    0    0    0    0    0    0
  603          -0.50           0.00    0    0    1   -1    1    0    0    1    0    0    0    0    0    0
  604           0.50           0.00    0    0    0    0    0    0    9  -12    0    0    0    0    0   -2
  605           0.00          -0.50    0    0    0    0    0    0    6   -9    0    0    0    0    0   -1
  606          -0.50           0.00    0    0    0    0    0    0    4   -4    0    0    0    0    0    1
  607          -0.50           0.00    0    0    0    0    0    0    0    6  -11    0    0    0    0   -2
  608           0.00           0.50    0    0    0    0    0    0    0    1   -3    0    0    0    0   -2
  609           0.40           0.10    0    0    2   -2    1   -1    0    2    0    0    0    0    0    0
  610          -0.40          -0.10    0    0    1   -1    1    0    0   -1    0    0    0    2    0    0
  611           0.40          -0.10    0    0    0    0    0    0    5   -6    0    0    0    0    0    2
  612          -0.40           0.10    0    0    0    0    0    0    5  -10    0    0    0    0    0   -2
  613           0.10           0.40    0    0    0    0    0    0    0    5    0   -3    0    0    0    2
  614           0.10           0.40    0    0    0    0    0    0    0    0    0    4    0    0    0    2
  615          -0.50           0.00    0    0    3    0    2    0    0    0    0    0    0    0    0    0
  616           0.00           0.50    2    0   -4    0   -1    0    0    0    0    0    0    0    0    0
  617           0.00          -0.50    2    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  618           0.00           0.50    0    0    2   -2    2    0   -3    3    0    0    0    0    0    0
  619           0.00           0.50    1    0    0   -1    2    0    0    0    0    0    0    0    0    0
  620           0.00          -0.50    1    1    0    0    2    0    0    0    0    0    0    0    0    0
  621           0.50           0.00    0    0    2   -2    2    0   -5    6    0    0    0    0    0    0
  622           0.00          -0.50    1    0    0    2   -2    0    0    0    0    0    0    0    0    0
  623           0.00           0.50    1    0   -2    2    0    0    0    0    0    0    0    0    0    0
  624           0.00          -0.40    1    0    2    2    0    0    0    0    0    0    0    0    0    0
  625          -0.20           0.20    0    0    0    0    2    0    0   -1    2    0    0    0    0    0
  626          -0.10           0.30    1    1    0    2    0    0    0    0    0    0    0    0    0    0
  627           0.00          -0.40    1    0    0    4    0    0    0    0    0    0    0    0    0    0
  628           0.40           0.00    1    0   -1    0    1    0    0    0    0    0    0    0    0    0
  629           0.00          -0.40    0    2    0   -2    0    0    0    0    0    0    0    0    0    0
  630           0.00          -0.40    1    0    0   -3    1    0    0    0    0    0    0    0    0    0
  631           0.00           0.40    4    0    0    0   -1    0    0    0    0    0    0    0    0    0
  632           0.00           0.40    1   -2    0   -2   -1    0    0    0    0    0    0    0    0    0
  633          -0.10           0.30    0    0    0    0    1    0    0   -8   15    0    0    0    0    0
  634           0.00           0.40    1    0    4   -4    2    0    0    0    0    0    0    0    0    0
  635           0.00          -0.40    1    1    0   -1    1    0    0    0    0    0    0    0    0    0
  636           0.00          -0.40    0    0    2    0    2    0    1   -1    0    0    0    0    0    0
  637           0.00          -0.40    1    1    0    2   -1    0    0    0    0    0    0    0    0    0
  638           0.00           0.40    2   -2    2    0    2    0    0    0    0    0    0    0    0    0
  639          -0.20          -0.20    0    0    1   -1    1    0    0   -1    0    0   -2    0    0    0
  640           0.20          -0.20    0    0    1   -1    1    0    0   -1    0   -2    4    0    0    0
  641           0.20           0.20    0    0    0    0    0    0    0    2    2    0    0    0    0    2
  642          -0.10           0.30    2   -2    2   -2    2    0    0    0    0    0    0    0    0    0
  643          -0.30           0.10    0    0    0    0    0    0    6  -10    0    0    0    0    0   -1
  644           0.10           0.30    0    0    0    0    0    0    0    6   -5    0    0    0    0    2
  645          -0.10           0.30    0    0    0    0    0    0    0    2    0   -2    0    0    0    2
  646           0.00          -0.40    5    0    2   -2    2    0    0    0    0    0    0    0    0    0
  647           0.00           0.40    3    0    2   -1    2    0    0    0    0    0    0    0    0    0
  648           0.00           0.40    1    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  649           0.00           0.40    1   -1   -2   -3   -2    0    0    0    0    0    0    0    0    0
  650           0.00           0.40    0    0    4   -1    2    0    0    0    0    0    0    0    0    0
  651           0.00          -0.40    0    0    2   -2    1    0    0   -2    0    3    0    0    0    0
  652           0.00          -0.40    0    0    2   -2    1    0    0   -2    0    0    2    0    0    0
  653           0.40           0.00    0    0    2   -2    1    0    0   -8   11    0    0    0    0    0
  654          -0.40           0.00    0    0    1   -1    1    0    0   -1    0   -1    2    0    0    0
  655          -0.40           0.00    0    0    0    0    0    0    8   -8    0    0    0    0    0   -1
  656          -0.40           0.00    0    0    0    0    0    0    8  -10    0    0    0    0    0   -1
  657           0.00           0.40    0    0    0    0    0    0    5   -5    0    0    0    0    0    2
  658           0.00          -0.40    0    0    0    0    0    0    5   -9    0    0    0    0    0   -2
  659           0.00          -0.40    0    0    0    0    0    0    4   -2    0    0    0    0    0    2
  660          -0.40           0.00    0    0    0    0    0    0    4   -2    0    0    0    0    0    1
  661           0.40           0.00    0    0    0    0    0    0    4   -3    0    0    0    0    0    2
  662           0.00          -0.40    0    0    0    0    0    0    3   -4    0    0    0    0    0   -1
  663           0.00           0.40    0    0    0    0    0    0    3   -6    0    0    0    0    0   -1
  664           0.00           0.40    0    0    0    0    0    0    2   -6    0    0    0    0    0   -2
  665           0.40           0.00    0    0    0    0    0    0    0    8  -15    0    0    0    0   -1
  666           0.00          -0.40    0    0    0    0    0    0    0    6    0    0    0    0    0    0
  667          -0.40           0.00    0    0    0    0    0    0    0    5   -2    0    0    0    0    2
  668           0.00           0.40    0    0    0    0    0    0    0    5   -3    0    0    0    0    2
  669           0.40           0.00    0    0    0    0    0    0    0    0    0    0    1    0    0    2
  670           0.00          -0.40    4    0    0    0    0    0    0    0    0    0    0    0    0    0
  671           0.00          -0.40    1   -2   -2    0   -2    0    0    0    0    0    0    0    0    0
  672           0.00          -0.40    1    2    2   -2    1    0    0    0    0    0    0    0    0    0
  673           0.00          -0.40    0    2    2    0    1    0    0    0    0    0    0    0    0    0
  674           0.00          -0.40    3    1    2    0    1    0    0    0    0    0    0    0    0    0
  675          -0.10           0.30    0    2   -2    0   -1    0    0    0    0    0    0    0    0    0
  676           0.00           0.40    1    1    2    1    1    0    0    0    0    0    0    0    0    0
  677           0.00           0.40    3   -1    2    0    1    0    0    0    0    0    0    0    0    0
  678           0.00           0.40    2    0    2   -1    1    0    0    0    0    0    0    0    0    0
  679           0.00          -0.40    2    0    0    2    2    0    0    0    0    0    0    0    0    0
  680           0.40           0.00    0    0    1   -1    2    0    0   -1    0    2    0    0    0    0
  681           0.00           0.40    0    0    2   -2   -2    0    0    0    0    0    0    0    0    0
  682           0.21           0.10    0    0    1   -1    2    0    0   -1    0   -2    5    0    0    0
  683           0.00           0.30    2    0   -4    2   -2    0    0    0    0    0    0    0    0    0
  684          -0.30           0.00    0    0    2   -2   -1    0   -5    6    0    0    0    0    0    0
  685           0.00          -0.30    2   -2    0   -2    1    0    0    0    0    0    0    0    0    0
  686           0.00           0.30    0    0    0    4    2    0    0    0    0    0    0    0    0    0
  687           0.00           0.30    2    0    0   -2   -1    0   -3    3    0    0    0    0    0    0
  688           0.20           0.10    0    0    1   -1   -1    0    0   -1    0   -1    0    0    0    0
  689           0.00          -0.30    1   -2    0   -2    1    0    0    0    0    0    0    0    0    0
  690           0.00           0.30    1   -2    0    0   -1    0    0    0    0    0    0    0    0    0
  691           0.30           0.00    0    0    0    0    1    0    0    0    0    0   -1    0    0    0
  692           0.00           0.30    2    0    0   -3   -1    0    0    0    0    0    0    0    0    0
  693           0.00          -0.30    2    1   -2    0    1    0    0    0    0    0    0    0    0    0
  694          -0.20           0.10    2    0    0   -2   -1    0    0   -6    8    0    0    0    0    0
  695          -0.10          -0.20    2    0   -1   -1    1    0    0    3   -7    0    0    0    0    0
  696           0.00          -0.30    2    1    2   -4    1    0    0    0    0    0    0    0    0    0
  697           0.10          -0.20    1    0    0   -1    1    0    0   -3    4    0    0    0    0    0
  698           0.00           0.30    2   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
  699           0.00           0.30    0    0    2    0    2    0   -1    1    0    0    0    0    0    0
  700           0.00          -0.30    0    0    2    0    2    0    0   -1    0    1    0    0    0    0
  701           0.00           0.30    0    0    2    0    2    0    0    1    0   -1    0    0    0    0
  702           0.00           0.30    0    1    0   -4    1    0    0    0    0    0    0    0    0    0
  703          -0.20           0.10    0    0    0    0    1    0    0   -9   17    0    0    0    0    0
  704           0.00          -0.30    1    1    4   -2    2    0    0    0    0    0    0    0    0    0
  705          -0.10          -0.20    2    0    2    0    1    0    0    1    0    0    0    0    0    0
  706          -0.10           0.20    1    0   -2    0   -2    0  -10    3    0    0    0    0    0    0
  707           0.20          -0.10    0    0    2   -2    1    0    0   -7    9    0    0    0    0    0
  708          -0.10          -0.20    0    0    0    0    0    0    2   -3    0    0    0    0    0    1
  709           0.20           0.10    0    0    0    0    0    0    1   -2    0    0    0    0    0   -2
  710           0.20          -0.10    0    0    0    0    0    0    1   -4    0    0    0    0    0   -2
  711          -0.20          -0.10    0    0    0    0    0    0    0    6  -10    0    0    0    0   -2
  712          -0.10          -0.20    0    0    0    0    0    0    0    5   -8    0    0    0    0   -2
  713           0.20          -0.10    0    0    0    0    0    0    0    3   -5    0    0    0    0   -1
  714           0.20           0.10    0    0    0    0    0    0    0    2   -6    0    0    0    0   -2
  715           0.00           0.30    4    0    2   -4    2    0    0    0    0    0    0    0    0    0
  716           0.00          -0.30    2    2    2   -2    2    0    0    0    0    0    0    0    0    0
  717           0.00           0.30    2    1    2    1    2    0    0    0    0    0    0    0    0    0
  718           0.00           0.30    2    0    0   -2   -1    0    0   -2    0    4   -5    0    0    0
  719           0.00          -0.30    2    0    0   -2   -2    0   -3    3    0    0    0    0    0    0
  720           0.30           0.00    2    0   -1   -1   -1    0    0   -1    0    0    0    0    0    0
  721           0.00          -0.30    1    0    2    3    2    0    0    0    0    0    0    0    0    0
  722           0.00           0.30    1    0    2   -2    2    0    0   -2    0    2    0    0    0    0
  723           0.30           0.00    1    0    1   -1    1    0    0   -1    0    0    0    0    0    0
  724           0.30           0.00    1    0   -1    1   -1    0  -18   17    0    0    0    0    0    0
  725           0.00          -0.30    1    0   -1   -1   -1    0   20  -20    0    0    0    0    0    0
  726           0.00          -0.30    1    0   -2   -2   -2    0   -3    3    0    0    0    0    0    0
  727           0.00          -0.30    0    2    2    2    2    0    0    0    0    0    0    0    0    0
  728           0.00           0.30    0    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  729           0.00           0.30    0    1    2    3    2    0    0    0    0    0    0    0    0    0
  730           0.00          -0.30    0    0    2   -2    1    0    0   -1    0    1    0    0    0    0
  731           0.30           0.00    0    0    1   -1    1    0    1   -2    0    0    0    0    0    0
  732           0.30           0.00    0    0    1   -1    1    0   -2    1    0    0    0    0    0    0
  733          -0.30           0.00    0    0    0    0    0    0    9  -11    0    0    0    0    0   -1
  734           0.00           0.30    0    0    0    0    0    0    8  -16    0    0    0    0    0   -2
  735          -0.30           0.00    0    0    0    0    0    0    5   -3    0    0    0    0    0    1
  736           0.30           0.00    0    0    0    0    0    0    0    6    0    0    0    0    0    1
  737          -0.30           0.00    0    0    0    0    0    0    0    6   -7    0    0    0    0    2
  738           0.00           0.30    0    0    0    0    0    0    0    4    0    0   -2    0    0    2
  739           0.30           0.00    0    0    0    0    0    0    0    4   -8    1    5    0    0   -2
  740          -0.30           0.00    0    0    0    0    0    0    0    3    0    0   -2    0    0    2
  741           0.30           0.00    0    0    0    0    0    0    0    1    0   -1    0    0    0    1
  742           0.30           0.00    0    0    0    0    0    0    0    1    0   -3    0    0    0   -1
  743           0.30           0.00    0    0    0    0    0    0    0    1   -6    0    0    0    0   -2
  744           0.00          -0.30    0    0    0    0    0    0    0    0    0    4   -5    0    0    2
  745           0.00           0.30    0    0    0    0    0    0    0    0    0    0    5    0    0    2
  746           0.00           0.30    0    0    0    0    0    0    0    0    0    0    0    2    0    2
  747           0.30           0.00    0    0    0    0    0    0    0    0    0    0    0    2    0    1
  748           0.00           0.30    0    0    0    0    0    0    0    0    0    0    0    0    2    2
  749           0.00          -0.30    1    0    0   -1   -1    0    0   -2    2    0    0    0    0    0
  750           0.00          -0.30    2   -1    2   -1    2    0    0    0    0    0    0    0    0    0
  751           0.30           0.00    2    0   -1   -1   -1    0    0   -1    0    2    0    0    0    0
  752           0.00           0.30    1    0    0   -6   -1    0    0    0    0    0    0    0    0    0
  753           0.00          -0.30    1   -1    0   -4   -1    0    0    0    0    0    0    0    0    0
  754           0.00           0.30    1    0    0    4   -1    0    0    0    0    0    0    0    0    0
  755           0.00          -0.30    3    0    0   -2    1    0    0    0    0    0    0    0    0    0
  756           0.30           0.00    0    0    0    0    1    0    5   -8    0    0    0    0    0    0
  757           0.00          -0.30    1   -2   -2   -2   -1    0    0    0    0    0    0    0    0    0
  758           0.00          -0.30    0    1    4   -2    1    0    0    0    0    0    0    0    0    0
  759           0.00          -0.30    2    0    4   -2    1    0    0    0    0    0    0    0    0    0
  760           0.00          -0.30    0    0    2    3    1    0    0    0    0    0    0    0    0    0
  761           0.30           0.00    0    0    1   -1    2    0    0   -1    0    1    0    0    0    0
  762           0.00           0.30    1   -2    2    0    1    0    0    0    0    0    0    0    0    0
  763           0.00          -0.30    2   -1    2   -2    1    0    0    0    0    0    0    0    0    0
  764           0.00           0.30    2    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
  765           0.00          -0.30    2    0   -4   -2   -1    0    0    0    0    0    0    0    0    0
  766           0.00          -0.30    2    0    2    1    1    0    0    0    0    0    0    0    0    0
  767           0.00           0.30    0    0    1   -1    2    0    0    0   -2    0    0    0    0    0
  768          -0.30           0.00    0    0    0    0    2    0    0    4   -8    3    0    0    0    0
  769          -0.30           0.00    0    0    0    0    2    0    0   -4    8   -3    0    0    0    0
  770           0.00           0.30    0    0    0    1   -2    0    0    0    0    0    0    0    0    0
  771           0.00          -0.30    0    0    2    0   -2    0    0    0    0    0    0    0    0    0
  772           0.00          -0.30    0    1    0   -2    2    0    0    0    0    0    0    0    0    0
  773          -0.20          -0.10    0    0    0    0    1    0    0    1    0   -2    0    0    0    0
  774           0.00          -0.30    0    0    0    0    1    0    2   -2    0    0    0    0    0    0
  775           0.00          -0.30    1    1    2   -4    2    0    0    0    0    0    0    0    0    0
  776           0.00          -0.21    0    1    2   -2    3    0    0    0    0    0    0    0    0    0
  777           0.00           0.20    2    0    2   -2    0    0    0    0    0    0    0    0    0    0
  778           0.00          -0.20    0    0    0    0    1    0   -2    2    0    0    0    0    0    0
  779           0.00          -0.20    3    0    2    0    0    0    0    0    0    0    0    0    0    0
  780           0.00          -0.20    0    0    0    0    1    0    0   -2    2    0    0    0    0    0
  781           0.00          -0.20    0    1    0   -4    0    0    0    0    0    0    0    0    0    0
  782           0.20           0.00    0    0    0    0    1    0   -3    4    0    0    0    0    0    0
  783           0.00          -0.20    3    0    0    2    0    0    0    0    0    0    0    0    0    0
  784           0.00           0.20    0    3    0    0    0    0    0    0    0    0    0    0    0    0
  785           0.00           0.20    2    0   -2   -2    0    0    0    0    0    0    0    0    0    0
  786           0.20           0.00    2    0    2   -4    0    0    0    0    0    0    0    0    0    0
  787           0.00           0.20    1    0    0   -3   -1    0    0    0    0    0    0    0    0    0
  788           0.00          -0.20    2    0    2   -2   -1    0    0    0    0    0    0    0    0    0
  789           0.00           0.20    1    0    2   -4   -1    0    0    0    0    0    0    0    0    0
  790           0.00           0.20    0    0    1   -1    2    0    0   -1    0    0    1    0    0    0
  791           0.00           0.20    1    0   -2    1    1    0    0    0    0    0    0    0    0    0
  792           0.00          -0.20    1    2    0   -2    0    0    0    0    0    0    0    0    0    0
  793           0.00          -0.20    2    0   -2   -2    1    0    0    0    0    0    0    0    0    0
  794           0.00          -0.20    1    0    0   -1    1    0    0   -1    0    1    0    0    0    0
  795           0.00          -0.20    2    0    0   -3    1    0    0    0    0    0    0    0    0    0
  796           0.00          -0.20    1    1    0    1    1    0    0    0    0    0    0    0    0    0
  797           0.00           0.20    1    1    0    1   -1    0    0    0    0    0    0    0    0    0
  798           0.00          -0.20    2    0    0   -1    1    0    0    0    0    0    0    0    0    0
  799           0.00           0.20    1    2    0    0    1    0    0    0    0    0    0    0    0    0
  800           0.00          -0.20    2    0    0   -2    1    0    0   -5    6    0    0    0    0    0
  801           0.00           0.20    1    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
  802           0.00          -0.20    1    0    0   -2    1    0    0   -2    0    2    0    0    0    0
  803           0.00           0.20    2    0   -4    2   -1    0    0    0    0    0    0    0    0    0
  804           0.20           0.00    0    0    2   -2    1    0    0    4   -8    3    0    0    0    0
  805           0.20           0.00    0    0    2   -2    1    0    0   -4    8   -3    0    0    0    0
  806           0.00           0.20    2   -1    0    2   -1    0    0    0    0    0    0    0    0    0
  807           0.00           0.20    1    0    0   -1   -1    0    0   -3    4    0    0    0    0    0
  808           0.10          -0.10    1   -2    0   -2    0    0    0    0    0    0    0    0    0    0
  809           0.00          -0.20    2    0    0   -2   -1    0    0   -2    0    3   -1    0    0    0
  810           0.00           0.20    1   -1   -4    2   -2    0    0    0    0    0    0    0    0    0
  811           0.00           0.20    3    0    0    2   -1    0    0    0    0    0    0    0    0    0
  812           0.00           0.20    1    0    0   -3    0    0    0    0    0    0    0    0    0    0
  813           0.00           0.20    2    0    2   -6    1    0    0    0    0    0    0    0    0    0
  814           0.00           0.20    2   -1   -2    2   -1    0    0    0    0    0    0    0    0    0
  815           0.00           0.20    1   -2   -2    2   -1    0    0    0    0    0    0    0    0    0
  816           0.00           0.20    2    0    4   -4    2    0    0    0    0    0    0    0    0    0
  817           0.20           0.00    2    0   -1   -1   -2    0    0   -1    0    2    0    0    0    0
  818           0.00          -0.20    4    1    2    0    2    0    0    0    0    0    0    0    0    0
  819           0.00           0.20    4   -1    2    0    2    0    0    0    0    0    0    0    0    0
  820           0.00          -0.20    4   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
  821           0.00          -0.20    3    0    4   -2    2    0    0    0    0    0    0    0    0    0
  822           0.00          -0.20    3    0    2    1    2    0    0    0    0    0    0    0    0    0
  823           0.00           0.20    3    0   -2    2   -1    0    0    0    0    0    0    0    0    0
  824           0.00          -0.20    3    0   -2   -1   -1    0    0    0    0    0    0    0    0    0
  825           0.00           0.20    2    2   -2   -4   -2    0    0    0    0    0    0    0    0    0
  826           0.00          -0.20    2    1    0   -4    0    0    0    0    0    0    0    0    0    0
  827           0.00           0.20    2    1   -2   -6   -2    0    0    0    0    0    0    0    0    0
  828           0.00           0.20    2    0    2   -3    2    0    0    0    0    0    0    0    0    0
  829           0.00           0.20    2   -2    2    2    2    0    0    0    0    0    0    0    0    0
  830           0.00          -0.20    1    1    2    4    2    0    0    0    0    0    0    0    0    0
  831           0.00          -0.20    1    1   -2   -3   -2    0    0    0    0    0    0    0    0    0
  832           0.00           0.20    1    1   -2   -6   -2    0    0    0    0    0    0    0    0    0
  833          -0.20           0.00    1    0    3    0    3    0    0    0    0    0    0    0    0    0
  834           0.20           0.00    1    0    1    1    1    0    0    1    0    0    0    0    0    0
  835           0.20           0.00    1    0   -1    1   -1    0    0    1    0    0    0    0    0    0
  836           0.00           0.20    1    0   -1    0   -1    0   -3    5    0    0    0    0    0    0
  837           0.20           0.00    1    0   -1   -2   -1    0    0    0    0    0    0    0    0    0
  838           0.00           0.20    1    0   -2   -2   -2    0    0   -2    0    3    0    0    0    0
  839           0.00          -0.20    1   -1    2    1    2    0    0    0    0    0    0    0    0    0
  840           0.00           0.20    1   -1   -2    1   -1    0    0    0    0    0    0    0    0    0
  841           0.00           0.20    0    3   -2   -2   -2    0    0    0    0    0    0    0    0    0
  842           0.00          -0.20    0    0    4    2    2    0    0    0    0    0    0    0    0    0
  843           0.00           0.20    0    0    2   -2    1    0    0    1    0   -1    0    0    0    0
  844           0.00           0.20    0    0    2   -2    1    0    0   -2    0    0    0    0    0    0
  845           0.00          -0.20    0    0    2   -2    1    0    0   -3    0    3    0    0    0    0
  846           0.00          -0.20    0    0    2   -2    1    0    0   -4    4    0    0    0    0    0
  847           0.00          -0.20    0    0    2   -2    1    0   -5    5    0    0    0    0    0    0
  848           0.00           0.20    0    0    1   -1    1    0    1   -3    0    0    0    0    0    0
  849           0.00          -0.20    0    0    1   -1    1    0    0    1   -4    0    0    0    0    0
  850           0.20           0.00    0    0    1   -1    1    0    0   -1    0    1   -3    0    0    0
  851           0.20           0.00    0    0    1   -1    1    0    0   -1    0    0    0   -1    0    0
  852           0.00          -0.20    0    0    1   -1    1    0    0   -4    6    0    0    0    0    0
  853           0.20           0.00    0    0    1   -1    1    0   -5    6    0    0    0    0    0    0
  854           0.00           0.20    0    0    0    0    0    0    8  -12    0    0    0    0    0   -2
  855           0.00          -0.20    0    0    0    0    0    0    7  -10    0    0    0    0    0   -1
  856           0.20           0.00    0    0    0    0    0    0    7  -11    0    0    0    0    0   -1
  857          -0.20           0.00    0    0    0    0    0    0    6   -4    0    0    0    0    0    1
  858          -0.20           0.00    0    0    0    0    0    0    6   -6    0    0    0    0    0    1
  859          -0.20           0.00    0    0    0    0    0    0    5   -5    0    0    0    0    0    1
  860           0.00          -0.20    0    0    0    0    0    0    5   -6    0    0    0    0    0   -1
  861          -0.20           0.00    0    0    0    0    0    0    4   -2    0    0    0    0    0    0
  862           0.00          -0.20    0    0    0    0    0    0    4   -5    0    0    0    0    0   -1
  863           0.20           0.00    0    0    0    0    0    0    4   -5    0    0    0    0    0   -2
  864           0.00          -0.20    0    0    0    0    0    0    3   -4    0    0    0    0    0    1
  865           0.20           0.00    0    0    0    0    0    0    3   -4    0    0    0    0    0   -2
  866          -0.20           0.00    0    0    0    0    0    0    3   -8    0    0    0    0    0   -2
  867           0.00           0.20    0    0    0    0    0    0    2   -5    0    0    0    0    0   -1
  868           0.00           0.20    0    0    0    0    0    0    1   -2    0    0    0    0    0   -1
  869           0.00           0.20    0    0    0    0    0    0    1   -4    0    0    0    0    0   -1
  870           0.00          -0.20    0    0    0    0    0    0    0    9  -17    0    0    0    0   -2
  871          -0.20           0.00    0    0    0    0    0    0    0    7   -7    0    0    0    0    2
  872          -0.20           0.00    0    0    0    0    0    0    0    7   -8    0    0    0    0    2
  873          -0.20           0.00    0    0    0    0    0    0    0    7   -9    0    0    0    0    2
  874          -0.20           0.00    0    0    0    0    0    0    0    5   -6    0    0    0    0    2
  875           0.20           0.00    0    0    0    0    0    0    0    5  -10    0    0    0    0   -2
  876           0.00           0.20    0    0    0    0    0    0    0    4    0   -4    0    0    0    2
  877           0.00          -0.20    0    0    0    0    0    0    0    4   -6    0    0    0    0   -2
  878           0.00          -0.20    0    0    0    0    0    0    0    4   -7    0    0    0    0   -1
  879           0.00          -0.20    0    0    0    0    0    0    0    4   -8    3    0    0    0    1
  880           0.00           0.20    0    0    0    0    0    0    0    4   -8    3    0    0    0   -1
  881           0.20           0.00    0    0    0    0    0    0    0    4   -8    1    5    0    0    2
  882           0.20           0.00    0    0    0    0    0    0    0    3    0    0    0    0    0    2
  883           0.00           0.20    0    0    0    0    0    0    0    3   -8    3    0    0    0   -2
  884           0.20           0.00    0    0    0    0    0    0    0    2    0    0   -2    0    0    1
  885           0.00          -0.20    0    0    0    0    0    0    0    2    0   -5    0    0    0   -2
  886           0.00          -0.20    0    0    0    0    0    0    0    2   -4    0    0    0    0    1
  887           0.20           0.00    0    0    0    0    0    0    0    2   -7    0    0    0    0   -2
  888           0.20           0.00    0    0    0    0    0    0    0    1    0    1    0    0    0    1
  889           0.00           0.20    0    0    0    0    0    0    0    1    0    0    0    0    0   -1
  890           0.20           0.00    0    0    0    0    0    0    0    1    0   -1    0    0    0   -1
  891           0.20           0.00    0    0    0    0    0    0    0    1    0   -2    5    0    0    2
  892           0.00           0.20    0    0    0    0    0    0    0    1    0   -2    0    0    0   -2
  893           0.00          -0.20    0    0    0    0    0    0    0    1    0   -5    0    0    0   -2
  894           0.10          -0.10    0    0    0    0    0    0    0    2    1    0    0    0    0    2
  895           0.00          -0.20    1    0    4    0    1    0    0    0    0    0    0    0    0    0
  896           0.00           0.20    4    0   -2    0   -1    0    0    0    0    0    0    0    0    0
  897           0.00          -0.20    0    1    0    4    1    0    0    0    0    0    0    0    0    0
  898           0.00           0.20    2    0    0    4    1    0    0    0    0    0    0    0    0    0
  899           0.00           0.20    0    0    0    0    1    0    0    7  -13    0    0    0    0    0
  900           0.00          -0.20    2    1    0    2    1    0    0    0    0    0    0    0    0    0
  901           0.00           0.20    1   -1    0    4    1    0    0    0    0    0    0    0    0    0
  902           0.20           0.00    0    0    2   -2    1    0   -8   11    0    0    0    0    0    0
  903           0.00           0.20    1    0    2   -6    1    0    0    0    0    0    0    0    0    0
  904           0.00           0.20    0    0    2   -2    2    0    0   -2    0    2    0    0    0    0
  905           0.00          -0.20    1    2    2    0    1    0    0    0    0    0    0    0    0    0
  906           0.00           0.20    3    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
  907           0.00           0.20    2    0    0   -6   -1    0    0    0    0    0    0    0    0    0
  908           0.00           0.20    1   -1    2    4    1    0    0    0    0    0    0    0    0    0
  909           0.00          -0.20    2    1    2    2    1    0    0    0    0    0    0    0    0    0
  910          -0.20           0.00    0    0    1    1    2    0    0    1    0    0    0    0    0    0
  911           0.00           0.20    2   -1   -2   -2   -1    0    0    0    0    0    0    0    0    0
  912           0.00          -0.20    0    1   -2    4   -1    0    0    0    0    0    0    0    0    0
  913           0.00           0.20    2    0    2    4    1    0    0    0    0    0    0    0    0    0
  914           0.00          -0.20    1   -1   -2   -4   -1    0    0    0    0    0    0    0    0    0
  915           0.00          -0.20    1    0   -4   -2   -1    0    0    0    0    0    0    0    0    0
  916           0.00           0.20    1   -2    2    2    1    0    0    0    0    0    0    0    0    0
  917           0.00           0.20    1    0   -2   -6   -1    0    0    0    0    0    0    0    0    0
  918           0.00          -0.20    3    1    2   -2    1    0    0    0    0    0    0    0    0    0
  919           0.00          -0.20    2   -2    0   -2   -2    0    0    0    0    0    0    0    0    0
  920           0.00           0.20    2   -1    0    2    1    0    0    0    0    0    0    0    0    0
  921           0.00           0.20    3    0    0    0    2    0    0    0    0    0    0    0    0    0
  922           0.00          -0.20    1   -1    2   -3    1    0    0    0    0    0    0    0    0    0
  923           0.00           0.20    0    0    1   -1    2    0   -5    7    0    0    0    0    0    0
  924           0.00          -0.20    2    1    0   -2   -2    0    0    0    0    0    0    0    0    0
  925          -0.20           0.00    0    0    1   -1    2    0    0   -1    0   -1    0    0    0    0
  926           0.00          -0.20    2    1   -2    0    0    0    0    0    0    0    0    0    0    0
  927           0.00           0.20    1    0   -2    0    2    0    0    0    0    0    0    0    0    0
  928           0.00          -0.20    1   -1    0    0   -2    0    0    0    0    0    0    0    0    0
  929           0.00          -0.20    0    2    0    0    2    0    0    0    0    0    0    0    0    0
  930          -0.20           0.00    0    0    0    0    1    0    3   -4    0    0    0    0    0    0
  931           0.00          -0.20    0    0    0    0    1    0    0    2   -2    0    0    0    0    0
  932           0.00           0.20    0    0    0    0    1    0    0    2    0   -2    0    0    0    0
  933           0.00          -0.20    0    1    2   -4    2    0    0    0    0    0    0    0    0    0
  934           0.00           0.20    0    0    1   -1    2    0   -8   12    0    0    0    0    0    0
  935           0.00           0.20    1    1   -2    2   -2    0    0    0    0    0    0    0    0    0
  936           0.00          -0.19    0    1    2   -2    0    0    0    0    0    0    0    0    0    0
  937           0.00           0.17    0    1   -2    2   -3    0    0    0    0    0    0    0    0    0
  938           0.00           0.11    1    0   -2    0   -3    0    0    0    0    0    0    0    0    0
  939           0.00          -0.10    1    1   -2   -2    0    0    0    0    0    0    0    0    0    0
  940           0.10           0.00    0    0    0    0    1    0    0   -1    0    2    0    0    0    0
  941           0.00          -0.10    0    1   -2   -2    0    0    0    0    0    0    0    0    0    0
  942           0.00           0.10    1    0    0    0   -1    0  -10    3    0    0    0    0    0    0
  943           0.00          -0.10    1    0    0    0    1    0  -10    3    0    0    0    0    0    0
  944           0.00          -0.10    1    0   -2   -4    0    0    0    0    0    0    0    0    0    0
  945           0.00          -0.10    2   -1    0    2    0    0    0    0    0    0    0    0    0    0
  946           0.00          -0.10    0    2    0    0   -2    0    0    0    0    0    0    0    0    0
  947           0.00          -0.10    1    0    0   -6    0    0    0    0    0    0    0    0    0    0
  948           0.00           0.10    0    1    4   -4    2    0    0    0    0    0    0    0    0    0
  949           0.00          -0.10    2    0    0   -6    0    0    0    0    0    0    0    0    0    0
  950          -0.10           0.00    0    0    0    0    2    0    0    0    0    1    0    0    0    0
  951           0.00          -0.10    3    0    0   -4    0    0    0    0    0    0    0    0    0    0
  952           0.00          -0.10    2   -1   -2    0    1    0    0    0    0    0    0    0    0    0
  953           0.00           0.10    2    2    0   -2   -1    0    0    0    0    0    0    0    0    0
  954           0.00          -0.10    0    0    0    3    2    0    0    0    0    0    0    0    0    0
  955           0.00          -0.10    3   -1    0    0    0    0    0    0    0    0    0    0    0    0
  956           0.10           0.00    0    0    2    0    2    0    2   -3    0    0    0    0    0    0
  957           0.10           0.00    0    0    2    0    2    0   -2    3    0    0    0    0    0    0
  958           0.00           0.10    0    0    2    0    2    0    2   -2    0    0    0    0    0    0
  959           0.00          -0.10    0    0    2    0    2    0   -2    2    0    0    0    0    0    0
  960           0.00          -0.10    2   -1   -2    0    0    0    0    0    0    0    0    0    0    0
  961           0.00           0.10    2    0   -2   -2   -2    0    0   -2    0    2    0    0    0    0
  962           0.00           0.10    1    2    2   -4    1    0    0    0    0    0    0    0    0    0];

%j = 1  Nb of terms = 277
b1=[...
  963      153041.82         878.89    0    0    0    0    1    0    0    0    0    0    0    0    0    0
  964       11714.49        -289.32    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
  965        2024.68         -50.99    0    0    2    0    2    0    0    0    0    0    0    0    0    0
  966       -1837.33          47.75    0    0    0    0    2    0    0    0    0    0    0    0    0    0
  967       -1312.21         -28.91    0    1    0    0    0    0    0    0    0    0    0    0    0    0
  968        -632.54           0.78    1    0    0    0    0    0    0    0    0    0    0    0    0    0
  969         459.68         -67.23    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
  970         344.50           1.46    0    0    2    0    1    0    0    0    0    0    0    0    0    0
  971         268.14          -7.03    1    0    2    0    2    0    0    0    0    0    0    0    0    0
  972         192.06          29.80    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
  973         139.64           0.15    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
  974        -113.94          -1.06    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
  975         109.81           3.18    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
  976         -56.37           0.13    0    0    0    2    0    0    0    0    0    0    0    0    0    0
  977         -56.17          -0.02    1    0    0    0    1    0    0    0    0    0    0    0    0    0
  978         -53.05          -1.23    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
  979         -51.60           0.17    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
  980          45.91          -0.11    1    0    2    0    1    0    0    0    0    0    0    0    0    0
  981         -42.45           0.02    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
  982          40.82          -1.03    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
  983          34.30          -1.24    0    0    2    2    2    0    0    0    0    0    0    0    0    0
  984          28.89           0.00    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
  985          27.61          -1.22    2    0    2    0    2    0    0    0    0    0    0    0    0    0
  986         -25.43           1.00    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
  987         -26.01           0.07    2    0    0    0    0    0    0    0    0    0    0    0    0    0
  988         -23.02           0.06    0    0    2    0    0    0    0    0    0    0    0    0    0    0
  989          19.37          -0.01    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
  990          14.05          -4.19    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
  991          18.18          -0.01    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
  992         -14.86          -0.09    0    2    0    0    0    0    0    0    0    0    0    0    0    0
  993          13.49          -0.01    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
  994          12.44          -0.27    0    1    0    0    1    0    0    0    0    0    0    0    0    0
  995          11.46           0.03    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
  996         -11.33          -0.06    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
  997          -9.81           0.01    2    0   -2    0    0    0    0    0    0    0    0    0    0    0
  998          -9.08          -0.02    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
  999           2.74          -4.56    0    0    1   -1    1    0    0   -1    0   -2    5    0    0    0
 1000           6.84          -0.04    1    0    2    2    2    0    0    0    0    0    0    0    0    0
 1001          -6.73           0.01    0    1    2    0    2    0    0    0    0    0    0    0    0    0
 1002           6.54           0.01    1    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1003          -6.35          -0.01    0    1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1004           5.90          -0.02    0    0    2    2    1    0    0    0    0    0    0    0    0    0
 1005          -5.85           0.02    1    0    0    2    0    0    0    0    0    0    0    0    0    0
 1006          -5.73           0.01    2    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1007           5.60           0.00    0    0    0    2    1    0    0    0    0    0    0    0    0    0
 1008          -5.16           0.00    1    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1009          -5.14           0.01    2    0    0   -2   -1    0    0    0    0    0    0    0    0    0
 1010           4.76          -0.02    2    0    2    0    1    0    0    0    0    0    0    0    0    0
 1011          -4.40           0.02    0    0    0    2   -1    0    0    0    0    0    0    0    0    0
 1012          -4.22           0.00    0    1   -2    2   -1    0    0    0    0    0    0    0    0    0
 1013          -4.20           0.01    1   -1    0    0    0    0    0    0    0    0    0    0    0    0
 1014           3.58           0.31    1    0    0   -1    0    0    0    0    0    0    0    0    0    0
 1015           3.87           0.01    0    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1016           3.76           0.00    0    0    0    1    0    0    0    0    0    0    0    0    0    0
 1017          -3.62          -0.01    2    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1018          -3.61           0.00    1    0   -2    0    0    0    0    0    0    0    0    0    0    0
 1019          -1.28          -2.14    0    0    0    0    0    0    0    0    0    2   -5    0    0   -1
 1020          -3.18           0.00    0    1    2   -2    1    0    0    0    0    0    0    0    0    0
 1021           3.01           0.00    1    1    0    0    0    0    0    0    0    0    0    0    0    0
 1022          -2.97           0.01    1    0    2    0    0    0    0    0    0    0    0    0    0    0
 1023           2.91           0.00    1   -1    0   -1    0    0    0    0    0    0    0    0    0    0
 1024          -2.73           0.00    2    0   -2    0   -2    0    0    0    0    0    0    0    0    0
 1025           2.58          -0.01    3    0    2    0    2    0    0    0    0    0    0    0    0    0
 1026           2.56          -0.01    1   -1    2    0    2    0    0    0    0    0    0    0    0    0
 1027          -2.51          -0.01    1    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1028          -2.35          -0.01    0    1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1029          -2.21           0.01    1    1    2    0    2    0    0    0    0    0    0    0    0    0
 1030          -2.04           0.01    2    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1031          -1.94           0.00    2    0    0    0    1    0    0    0    0    0    0    0    0    0
 1032           0.41          -1.43    0    0    0    0    0    0    0    4   -8    3    0    0    0    0
 1033          -1.84           0.00    0    0    0    0    0    0    3   -5    0    0    0    0    0   -2
 1034          -1.77           0.01    1    0   -2    2   -1    0    0    0    0    0    0    0    0    0
 1035           0.00           1.77    0    1   -1    1   -1    0    0    0    0    0    0    0    0    0
 1036           1.76           0.00    1    0    0    0    2    0    0    0    0    0    0    0    0    0
 1037          -1.07          -0.53    0    0    1   -1    1    0   -8   12    0    0    0    0    0    0
 1038          -1.48           0.00    0    0    2    1    2    0    0    0    0    0    0    0    0    0
 1039          -1.40           0.01    3    0    0    0    0    0    0    0    0    0    0    0    0    0
 1040          -1.35          -0.01    1    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1041          -1.32           0.00    0    0    0    0    0    0    1   -1    0    0    0    0    0    0
 1042          -1.28           0.00    0    0    0    0    0    0    0    8  -16    4    5    0    0    0
 1043           1.24           0.00    1    0    0    0   -2    0    0    0    0    0    0    0    0    0
 1044           1.23           0.00    2    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1045           1.19           0.00    1    0    0   -4    0    0    0    0    0    0    0    0    0    0
 1046           1.18          -0.01    1    0    2    2    1    0    0    0    0    0    0    0    0    0
 1047           1.17           0.00    1   -1    0   -1   -1    0    0    0    0    0    0    0    0    0
 1048          -1.15           0.00    1    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1049           1.14           0.00    2    0    0   -4    0    0    0    0    0    0    0    0    0    0
 1050          -1.14           0.00    0    2   -2    2   -1    0    0    0    0    0    0    0    0    0
 1051           1.09           0.03    0    0    0    0    0    0    0    1    0   -1    0    0    0    0
 1052          -1.08           0.00    2    0   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1053           1.04           0.00    0    0    0    0    0    0    0    0    0    2    0    0    0    2
 1054           1.02           0.00    1    0   -4    0   -2    0    0    0    0    0    0    0    0    0
 1055           0.98          -0.01    2    0    2    2    2    0    0    0    0    0    0    0    0    0
 1056           0.91           0.02    1    0    0   -1   -1    0    0    0    0    0    0    0    0    0
 1057           0.00           0.93    1    0   -1    0   -1    0    0    0    0    0    0    0    0    0
 1058          -0.91           0.00    2    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1059          -0.90           0.00    2    1    0   -2    0    0    0    0    0    0    0    0    0    0
 1060           0.86           0.00    1    0    0    2    1    0    0    0    0    0    0    0    0    0
 1061          -0.84           0.00    1   -1    0   -2    0    0    0    0    0    0    0    0    0    0
 1062          -0.83           0.00    3    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1063          -0.82           0.00    0    0    4   -2    2    0    0    0    0    0    0    0    0    0
 1064           0.41           0.39    0    0    0    0    1    0    0   -1    2    0    0    0    0    0
 1065           0.40          -0.38    0    0    0    0    0    0    0    1   -2    0    0    0    0    0
 1066           0.78           0.00    0    1   -2    2    0    0    0    0    0    0    0    0    0    0
 1067           0.74           0.00    0    0    2   -2   -1    0    0    0    0    0    0    0    0    0
 1068          -0.73           0.00    0    1    2    0    1    0    0    0    0    0    0    0    0    0
 1069           0.68           0.00    1    0    2   -2    0    0    0    0    0    0    0    0    0    0
 1070           0.66           0.00    1    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1071          -0.64           0.00    2    0   -2    0    1    0    0    0    0    0    0    0    0    0
 1072          -0.63           0.00    0    1    0    0    2    0    0    0    0    0    0    0    0    0
 1073           0.63           0.00    0    0    2   -1    2    0    0    0    0    0    0    0    0    0
 1074           0.62           0.00    0    0    2    4    2    0    0    0    0    0    0    0    0    0
 1075           0.60           0.00    0    1    0    2    0    0    0    0    0    0    0    0    0    0
 1076          -0.59           0.00    0    0    2    0   -1    0    0    0    0    0    0    0    0    0
 1077          -0.59           0.00    0    1   -2    0   -1    0    0    0    0    0    0    0    0    0
 1078           0.59           0.00    0    1    2   -2    0    0    0    0    0    0    0    0    0    0
 1079           0.57           0.00    1    0   -2   -2    0    0    0    0    0    0    0    0    0    0
 1080           0.38          -0.19    0    0    0    0    0    0    8  -13    0    0    0    0    0   -1
 1081          -0.01          -0.55    0    0    0    0    0    0    2   -3    0    0    0    0    0    0
 1082           0.44          -0.11    0    0    0    0    0    0    0    0    0    2   -5    0    0    0
 1083           0.53           0.00    0    0    0    0    0    0    2   -2    0    0    0    0    0    0
 1084          -0.53           0.00    1   -1    0   -1   -2    0    0    0    0    0    0    0    0    0
 1085           0.52           0.00    1   -1    2    2    2    0    0    0    0    0    0    0    0    0
 1086          -0.52           0.00    2    0    0    2    0    0    0    0    0    0    0    0    0    0
 1087           0.53           0.00    0    1    0    0   -2    0    0    0    0    0    0    0    0    0
 1088           0.52           0.00    1    1    0   -2    1    0    0    0    0    0    0    0    0    0
 1089           0.51           0.00    1    0   -2    2    0    0    0    0    0    0    0    0    0    0
 1090           0.51           0.00    1   -1   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1091          -0.21          -0.30    0    0    0    0    0    0    8  -13    0    0    0    0    0    0
 1092          -0.50           0.00    0    1    0    1    0    0    0    0    0    0    0    0    0    0
 1093          -0.11           0.37    0    0    0    0    0    0    0    2   -8    3    0    0    0   -2
 1094          -0.11           0.37    0    0    0    0    0    0    0    6   -8    3    0    0    0    2
 1095          -0.48           0.00    0    1    2    2    2    0    0    0    0    0    0    0    0    0
 1096          -0.46          -0.01    0    0    0    0    0    0    0    3    0   -1    0    0    0    2
 1097          -0.47           0.00    1   -1    0    0    1    0    0    0    0    0    0    0    0    0
 1098          -0.03           0.43    0    0    1   -1    1    0    0   -1    0    2   -5    0    0    0
 1099           0.45           0.00    3    0    2    0    1    0    0    0    0    0    0    0    0    0
 1100          -0.44           0.00    0    0    0    4    0    0    0    0    0    0    0    0    0    0
 1101          -0.44           0.00    1    0    0    2   -1    0    0    0    0    0    0    0    0    0
 1102          -0.44           0.00    1   -1    0    2    0    0    0    0    0    0    0    0    0    0
 1103           0.43           0.00    2   -1    2    0    2    0    0    0    0    0    0    0    0    0
 1104           0.44           0.00    0    0    0    0    0    0    4   -6    0    0    0    0    0   -2
 1105           0.42           0.00    0    0    0    2    2    0    0    0    0    0    0    0    0    0
 1106          -0.42           0.00    1    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1107           0.41           0.00    1   -1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1108          -0.41           0.00    0    0    0    0    0    0    2   -4    0    0    0    0    0   -2
 1109           0.02           0.39    0    0    2   -2    1    0   -5    6    0    0    0    0    0    0
 1110           0.40           0.00    1    0    2   -4    1    0    0    0    0    0    0    0    0    0
 1111          -0.40           0.00    0    1   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1112          -0.39           0.00    2    0    0   -2    0    0    0   -2    0    2    0    0    0    0
 1113           0.39           0.00    0    3    2   -2    2    0    0    0    0    0    0    0    0    0
 1114           0.15          -0.24    0    0    0    0    0    0    0    1    0   -2    0    0    0    0
 1115          -0.37          -0.01    0    0    0    0    0    0    0    2    0   -2    0    0    0    0
 1116           0.37           0.00    1   -1    2    0    1    0    0    0    0    0    0    0    0    0
 1117          -0.37           0.00    0    0    2    2    0    0    0    0    0    0    0    0    0    0
 1118          -0.37           0.00    2    1    2    0    2    0    0    0    0    0    0    0    0    0
 1119          -0.31           0.06    2    0    0   -2    0    0    0   -2    0    3    0    0    0    0
 1120          -0.35           0.00    1    0    0   -2   -2    0    0    0    0    0    0    0    0    0
 1121           0.35           0.00    0    0    0    1    1    0    0    0    0    0    0    0    0    0
 1122          -0.07          -0.27    0    0    0    0    1    0    0   -4    8   -3    0    0    0    0
 1123          -0.33           0.01    0    0    0    0    0    0    2    0    0    0    0    0    0    2
 1124          -0.33           0.00    2   -1    0    0    0    0    0    0    0    0    0    0    0    0
 1125           0.07          -0.26    0    0    0    0    1    0    0    4   -8    3    0    0    0    0
 1126           0.33           0.00    0    0    0    0    0    0    0    2   -2    0    0    0    0    0
 1127           0.00          -0.32    1    0   -1    0   -2    0    0    0    0    0    0    0    0    0
 1128           0.32           0.00    1    1    0    0    1    0    0    0    0    0    0    0    0    0
 1129          -0.32           0.00    1    1    2    0    1    0    0    0    0    0    0    0    0    0
 1130           0.32           0.00    1    0   -2    2   -2    0    0    0    0    0    0    0    0    0
 1131          -0.24          -0.07    0    0    1   -1    1    0    0    0   -2    0    0    0    0    0
 1132           0.24           0.07    0    0    1   -1    0    0    0    0   -2    0    0    0    0    0
 1133           0.30           0.00    0    0    0    0    0    0    1    1    0    0    0    0    0    2
 1134           0.08          -0.22    0    0    0    0    0    0    0    0    0    1    0    0    0    0
 1135          -0.30           0.00    1    0    2    1    2    0    0    0    0    0    0    0    0    0
 1136          -0.30           0.00    2    0    2    0    0    0    0    0    0    0    0    0    0    0
 1137           0.30           0.00    0    1    0   -2    1    0    0    0    0    0    0    0    0    0
 1138           0.30           0.00    1    0    2   -1    2    0    0    0    0    0    0    0    0    0
 1139           0.00          -0.29    0    0    0    0    0    0    3   -4    0    0    0    0    0    0
 1140           0.00          -0.29    1    0   -1    0    0    0    0    0    0    0    0    0    0    0
 1141           0.20          -0.09    1    0    0    0    0    0  -18   16    0    0    0    0    0    0
 1142           0.29           0.00    1    0    0    1    0    0    0    0    0    0    0    0    0    0
 1143          -0.05          -0.24    0    0    0    0    0    0    0    0    0    2    0    0    0    1
 1144           0.29           0.00    0    0    0    1   -1    0    0    0    0    0    0    0    0    0
 1145          -0.27           0.00    1    0    0   -2    2    0    0    0    0    0    0    0    0    0
 1146          -0.19          -0.08    1    0    0    0    0    0  -10    3    0    0    0    0    0    0
 1147          -0.27           0.00    1   -1    0    0   -1    0    0    0    0    0    0    0    0    0
 1148           0.25           0.00    2    1    0    0    0    0    0    0    0    0    0    0    0    0
 1149           0.25           0.00    2    0    0   -2   -1    0    0   -2    0    2    0    0    0    0
 1150          -0.25           0.00    0    0    2    1    1    0    0    0    0    0    0    0    0    0
 1151           0.25           0.00    1    2    0   -2    0    0    0    0    0    0    0    0    0    0
 1152          -0.25           0.00    0    3    0    0    0    0    0    0    0    0    0    0    0    0
 1153          -0.01           0.23    0    0    0    0    0    0    5   -8    0    0    0    0    0   -2
 1154          -0.23           0.00    1    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
 1155           0.23           0.00    0    0    0    0    0    0    0    1    0    1    0    0    0    2
 1156           0.23           0.00    4    0    2    0    2    0    0    0    0    0    0    0    0    0
 1157          -0.15          -0.07    0    0    1   -1    1    0    0   -1    0    0   -1    0    0    0
 1158          -0.23           0.00    1    0   -2    0    1    0    0    0    0    0    0    0    0    0
 1159          -0.22           0.00    2    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1160           0.22           0.00    0    1    2    1    2    0    0    0    0    0    0    0    0    0
 1161          -0.22           0.00    1    1    2   -2    1    0    0    0    0    0    0    0    0    0
 1162          -0.22           0.00    1    0    4   -2    2    0    0    0    0    0    0    0    0    0
 1163           0.04          -0.17    0    0    1   -1    1    0    0   -1    0   -1    0    0    0    0
 1164          -0.01          -0.21    0    0    2   -2    0    0   -5    6    0    0    0    0    0    0
 1165           0.08          -0.14    0    0    0    0    0    0    0    2   -4    0    0    0    0    0
 1166          -0.01           0.19    0    0    0    0    0    0    0    0    0    2   -5    0    0    1
 1167           0.21           0.00    2    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1168          -0.20           0.00    1    0   -2    1   -1    0    0    0    0    0    0    0    0    0
 1169          -0.20           0.00    2   -2    0   -2    0    0    0    0    0    0    0    0    0    0
 1170          -0.04          -0.16    0    0    0    0    0    0    8  -13    0    0    0    0    0   -2
 1171           0.19           0.00    0    2    0   -2    0    0    0    0    0    0    0    0    0    0
 1172           0.19           0.00    1    1    0    0   -1    0    0    0    0    0    0    0    0    0
 1173          -0.19           0.00    2    0   -2   -4   -1    0    0    0    0    0    0    0    0    0
 1174           0.18           0.00    0    0    0    0    0    0    5   -7    0    0    0    0    0   -2
 1175          -0.18           0.00    0    1    0    2    1    0    0    0    0    0    0    0    0    0
 1176           0.18           0.00    1    0   -4    0   -1    0    0    0    0    0    0    0    0    0
 1177           0.17           0.00    2    0    2    2    1    0    0    0    0    0    0    0    0    0
 1178          -0.12           0.06    1    0    2    0    2    0    0    1    0    0    0    0    0    0
 1179           0.13          -0.04    0    0    0    0    0    0    3   -5    0    0    0    0    0    0
 1180          -0.11           0.06    1    0   -2    0   -2    0    0    4   -8    3    0    0    0    0
 1181           0.17           0.00    0    0    2   -3    2    0    0    0    0    0    0    0    0    0
 1182           0.16           0.00    0    0    0    0    0    0    0    4    0   -2    0    0    0    2
 1183          -0.17           0.00    0    0    4    0    2    0    0    0    0    0    0    0    0    0
 1184          -0.17           0.00    1    0    0   -4   -1    0    0    0    0    0    0    0    0    0
 1185          -0.14           0.02    1    0    0   -1    1    0    0    0    0    0    0    0    0    0
 1186           0.14           0.03    0    0    0    0    0    0    0    2    0   -1    0    0    0    2
 1187           0.00           0.15    0    0    0    0    0    0    2   -1    0    0    0    0    0    2
 1188          -0.15           0.00    1    1   -2   -4   -2    0    0    0    0    0    0    0    0    0
 1189          -0.14           0.01    0    0    0    0    0    0    0    2    0    1    0    0    0    2
 1190           0.16           0.00    2    0    0   -2   -2    0    0    0    0    0    0    0    0    0
 1191          -0.06           0.10    0    0    0    0    0    0    8  -11    0    0    0    0    0   -2
 1192           0.05           0.10    0    0    0    0    0    0    0    8  -16    4    5    0    0   -2
 1193           0.02           0.13    0    0    1   -1    1    0    0   -1    0    2    0    0    0    0
 1194          -0.11           0.04    0    0    0    0    0    0    0    8  -16    4    5    0    0    2
 1195          -0.12          -0.02    0    0    1   -1    1    0   -5    7    0    0    0    0    0    0
 1196          -0.05          -0.10    0    0    0    0    0    0    0    0    0    0    2    0    0    1
 1197           0.14           0.00    1    0    2    4    2    0    0    0    0    0    0    0    0    0
 1198          -0.09           0.05    1    0    0   -2    0    0   19  -21    3    0    0    0    0    0
 1199           0.00           0.14    0    0    1    0    1    0    0    0    0    0    0    0    0    0
 1200           0.14           0.00    3    0    0   -4    0    0    0    0    0    0    0    0    0    0
 1201          -0.14           0.00    1    1   -2    0   -2    0    0    0    0    0    0    0    0    0
 1202           0.04           0.10    0    0    1   -1    1    0    0   -1    0    0    2    0    0    0
 1203          -0.06           0.08    0    0    0    0    0    0    0    3    0   -2    0    0    0    2
 1204           0.05           0.09    0    0    0    0    0    0    8  -15    0    0    0    0    0   -2
 1205          -0.14           0.00    0    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1206           0.08           0.06    0    0    0    0    0    0    0    1    2    0    0    0    0    2
 1207           0.14           0.00    0    0    0    4    1    0    0    0    0    0    0    0    0    0
 1208           0.14           0.00    0    0    2   -4    1    0    0    0    0    0    0    0    0    0
 1209           0.13           0.00    1    1    0   -4    0    0    0    0    0    0    0    0    0    0
 1210          -0.07           0.06    0    0    0    0    0    0    0    3   -2    0    0    0    0    2
 1211           0.11          -0.02    0    0    0    0    0    0    0    0    0    3    0    0    0    2
 1212          -0.13           0.00    3    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1213          -0.13           0.00    0    0    0    0    0    0    0    4   -2    0    0    0    0    2
 1214          -0.13           0.00    1    1    2    2    2    0    0    0    0    0    0    0    0    0
 1215          -0.13           0.00    0    0    4   -2    1    0    0    0    0    0    0    0    0    0
 1216          -0.12           0.00    2    0    0   -2    0    0   -3    3    0    0    0    0    0    0
 1217           0.12           0.00    3    0    2    2    2    0    0    0    0    0    0    0    0    0
 1218           0.12           0.00    0    0    0    0    0    0    0    0    0    0    2    0    0    2
 1219          -0.12           0.00    2    1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1220           0.00          -0.12    1    0    0   -1    0    0   -3    4    0    0    0    0    0    0
 1221          -0.02          -0.09    0    0    0    0    0    0    4   -6    0    0    0    0    0   -1
 1222           0.02          -0.09    0    0    0    0    0    0    1   -1    0    0    0    0    0   -1
 1223          -0.11           0.00    0    2   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1224           0.11           0.00    2    0    0   -4   -1    0    0    0    0    0    0    0    0    0
 1225           0.07          -0.04    0    0    0    0    0    0    0    3   -4    0    0    0    0    0
 1226           0.11           0.00    0    0    2    4    1    0    0    0    0    0    0    0    0    0
 1227           0.11           0.00    0    0    0    2   -2    0    0    0    0    0    0    0    0    0
 1228          -0.11           0.00    3    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1229           0.10           0.00    0    0    0    0    0    0    0    1    0   -3    0    0    0   -2
 1230          -0.10           0.00    0    0    2   -2    1    0   -3    3    0    0    0    0    0    0
 1231           0.10           0.00    0    0    0    0    0    0    6   -8    0    0    0    0    0   -2
 1232           0.10           0.00    2    1    0   -4    0    0    0    0    0    0    0    0    0    0
 1233           0.10           0.00    0    0    1   -1    1    0    0    3   -8    3    0    0    0    0
 1234           0.00           0.10    0    0    0    0    0    0    3   -2    0    0    0    0    0    2
 1235           0.00           0.10    0    0    0    0    0    0    2   -5    0    0    0    0    0   -2
 1236          -0.10           0.00    4    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1237           0.10           0.00    2    0    0    2    1    0    0    0    0    0    0    0    0    0
 1238          -0.10           0.00    1   -1    0   -2   -1    0    0    0    0    0    0    0    0    0
 1239           0.10           0.00    0    0    0    0    0    0    1   -3    0    0    0    0    0   -2];

%j = 2  Nb of terms = 30
b2=[...
 1240         121.15       -2301.27    0    0    0    0    1    0    0    0    0    0    0    0    0    0
 1241          -0.98        -143.27    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1242          -0.27         -24.46    0    0    2    0    2    0    0    0    0    0    0    0    0    0
 1243           0.24          22.41    0    0    0    0    2    0    0    0    0    0    0    0    0    0
 1244          -1.19          -5.61    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
 1245           3.57          -1.83    0    1    0    0    0    0    0    0    0    0    0    0    0    0
 1246           0.24          -5.02    0    0    2    0    1    0    0    0    0    0    0    0    0    0
 1247          -0.04          -3.23    1    0    2    0    2    0    0    0    0    0    0    0    0    0
 1248          -0.48           2.40    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
 1249          -0.10           1.73    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
 1250          -0.01           1.33    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
 1251          -0.04           0.83    1    0    0    0    1    0    0    0    0    0    0    0    0    0
 1252          -0.05          -0.79    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
 1253           0.03          -0.66    1    0    2    0    1    0    0    0    0    0    0    0    0    0
 1254           0.00          -0.64    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
 1255           0.04           0.61    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1256          -0.01          -0.41    0    0    2    2    2    0    0    0    0    0    0    0    0    0
 1257          -0.01           0.35    0    2   -2    2   -2    0    0    0    0    0    0    0    0    0
 1258          -0.01          -0.33    2    0    2    0    2    0    0    0    0    0    0    0    0    0
 1259           0.01           0.31    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1260           0.01           0.27    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0
 1261          -0.07          -0.17    0    2    2   -2    2    0    0    0    0    0    0    0    0    0
 1262           0.07           0.17    1    0    0    0    0    0    0    0    0    0    0    0    0    0
 1263           0.02          -0.21    0    1    0    0    1    0    0    0    0    0    0    0    0    0
 1264           0.01           0.20    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
 1265           0.01          -0.17    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
 1266           0.01          -0.16    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
 1267           0.00          -0.13    1    0   -2   -2   -1    0    0    0    0    0    0    0    0    0
 1268          -0.07          -0.04    0    0    1   -1    1    0    0   -1    0   -2    5    0    0    0
 1269           0.02           0.08    0    1    2    0    2    0    0    0    0    0    0    0    0    0];
 
%j = 3  Nb of terms = 5
b3=[...
 1270         -15.23          -1.62    0    0    0    0    1    0    0    0    0    0    0    0    0    0
 1271          -1.16          -0.01    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
 1272          -0.20           0.00    0    0    2    0    2    0    0    0    0    0    0    0    0    0
 1273           0.18           0.00    0    0    0    0    2    0    0    0    0    0    0    0    0    0
 1274           0.13           0.00    0    1    0    0    0    0    0    0    0    0    0    0    0    0];
 
%j = 4  Nb of terms = 1
b4=[...
 1275          -0.01           0.11    0    0    0    0    1    0    0    0    0    0    0    0    0    0];

 
% *  ---------------------------------------
% *  The series for s+XY/2: numerical values
% *  ---------------------------------------
 
% *  Polynomial coefficients [as]
      Spol=[      94   ,...
                3808.35,...
                -119.94,...
              -72574.09,...
                  27.70,...
                  15.61 ];

% Expression for the quantity s(t)+XY/2 based on the IAU2000A precession-nutation 
% model 
% 
% 
% Updated Table (12/11/2003) for ensuring continuity of UT1 on 1st January 2003
% 
% --------------------------------------------------------------------------------------------------------------
% 
% s + XY/2 = polynomial part + non-polynomial part
% 
% Non-polynomial part (unit microarcsecond)
% (ARG being for various combination of the fundamental arguments of the nutation theory)
% 
%   Sum_i[C_{s,0})_i * sin(ARG) + C_{c,0})_i * cos(ARG)] 
% 
% + Sum_i)j=1,4 [C_{s,j})_i * t^j * sin(ARG) + C_{c,j})_i * cos(ARG)] * t^j]
% 
% The Table below provides the values for C_{s,j})_i and C_{c,j})_i
% 
% Cutoff (0.1 microarcsecond and periods less than 500 years)
% 
% The expressions for the fundamental arguments appearing in columns 4 to 8 (luni-solar part)
% and in columns 6 to 17 (planetary part) are those of the IERS Conventions 2000
% 
% --------------------------------------------------------------------------------------------------------------
% 
%     i    C_{s,j})_i      C_{c,j})_i    l    l'   F    D   Om L_Me L_Ve  L_E L_Ma  L_J L_Sa  L_U L_Ne  p_A
% 
% --------------------------------------------------------------------------------------------------------------
 
%j = 0  Nb of terms = 33
c0=[...
    1       -2640.73           0.39    0    0    0    0    1    0    0    0    0    0    0    0    0    0
    2         -63.53           0.02    0    0    0    0    2    0    0    0    0    0    0    0    0    0
    3         -11.75          -0.01    0    0    2   -2    3    0    0    0    0    0    0    0    0    0
    4         -11.21          -0.01    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
    5           4.57           0.00    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
    6          -2.02           0.00    0    0    2    0    3    0    0    0    0    0    0    0    0    0
    7          -1.98           0.00    0    0    2    0    1    0    0    0    0    0    0    0    0    0
    8           1.72           0.00    0    0    0    0    3    0    0    0    0    0    0    0    0    0
    9           1.41           0.01    0    1    0    0    1    0    0    0    0    0    0    0    0    0
   10           1.26           0.01    0    1    0    0   -1    0    0    0    0    0    0    0    0    0
   11           0.63           0.00    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
   12           0.63           0.00    1    0    0    0    1    0    0    0    0    0    0    0    0    0
   13          -0.46           0.00    0    1    2   -2    3    0    0    0    0    0    0    0    0    0
   14          -0.45           0.00    0    1    2   -2    1    0    0    0    0    0    0    0    0    0
   15          -0.36           0.00    0    0    4   -4    4    0    0    0    0    0    0    0    0    0
   16           0.24           0.12    0    0    1   -1    1    0   -8   12    0    0    0    0    0    0
   17          -0.32           0.00    0    0    2    0    0    0    0    0    0    0    0    0    0    0
   18          -0.28           0.00    0    0    2    0    2    0    0    0    0    0    0    0    0    0
   19          -0.27           0.00    1    0    2    0    3    0    0    0    0    0    0    0    0    0
   20          -0.26           0.00    1    0    2    0    1    0    0    0    0    0    0    0    0    0
   21           0.21           0.00    0    0    2   -2    0    0    0    0    0    0    0    0    0    0
   22          -0.19           0.00    0    1   -2    2   -3    0    0    0    0    0    0    0    0    0
   23          -0.18           0.00    0    1   -2    2   -1    0    0    0    0    0    0    0    0    0
   24           0.10          -0.05    0    0    0    0    0    0    8  -13    0    0    0    0    0   -1
   25          -0.15           0.00    0    0    0    2    0    0    0    0    0    0    0    0    0    0
   26           0.14           0.00    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   27           0.14           0.00    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
   28          -0.14           0.00    1    0    0   -2    1    0    0    0    0    0    0    0    0    0
   29          -0.14           0.00    1    0    0   -2   -1    0    0    0    0    0    0    0    0    0
   30          -0.13           0.00    0    0    4   -2    4    0    0    0    0    0    0    0    0    0
   31           0.11           0.00    0    0    2   -2    4    0    0    0    0    0    0    0    0    0
   32          -0.11           0.00    1    0   -2    0   -3    0    0    0    0    0    0    0    0    0
   33          -0.11           0.00    1    0   -2    0   -1    0    0    0    0    0    0    0    0    0];
 
%j = 1  Nb of terms = 3
c1=[...
   34          -0.07           3.57    0    0    0    0    2    0    0    0    0    0    0    0    0    0
   35           1.71          -0.03    0    0    0    0    1    0    0    0    0    0    0    0    0    0
   36           0.00           0.48    0    0    2   -2    3    0    0    0    0    0    0    0    0    0];

%j = 2  Nb of terms = 25
c2=[...      
   37         743.53          -0.17    0    0    0    0    1    0    0    0    0    0    0    0    0    0
   38          56.91           0.06    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
   39           9.84          -0.01    0    0    2    0    2    0    0    0    0    0    0    0    0    0
   40          -8.85           0.01    0    0    0    0    2    0    0    0    0    0    0    0    0    0
   41          -6.38          -0.05    0    1    0    0    0    0    0    0    0    0    0    0    0    0
   42          -3.07           0.00    1    0    0    0    0    0    0    0    0    0    0    0    0    0
   43           2.23           0.00    0    1    2   -2    2    0    0    0    0    0    0    0    0    0
   44           1.67           0.00    0    0    2    0    1    0    0    0    0    0    0    0    0    0
   45           1.30           0.00    1    0    2    0    2    0    0    0    0    0    0    0    0    0
   46           0.93           0.00    0    1   -2    2   -2    0    0    0    0    0    0    0    0    0
   47           0.68           0.00    1    0    0   -2    0    0    0    0    0    0    0    0    0    0
   48          -0.55           0.00    0    0    2   -2    1    0    0    0    0    0    0    0    0    0
   49           0.53           0.00    1    0   -2    0   -2    0    0    0    0    0    0    0    0    0
   50          -0.27           0.00    0    0    0    2    0    0    0    0    0    0    0    0    0    0
   51          -0.27           0.00    1    0    0    0    1    0    0    0    0    0    0    0    0    0
   52          -0.26           0.00    1    0   -2   -2   -2    0    0    0    0    0    0    0    0    0
   53          -0.25           0.00    1    0    0    0   -1    0    0    0    0    0    0    0    0    0
   54           0.22           0.00    1    0    2    0    1    0    0    0    0    0    0    0    0    0
   55          -0.21           0.00    2    0    0   -2    0    0    0    0    0    0    0    0    0    0
   56           0.20           0.00    2    0   -2    0   -1    0    0    0    0    0    0    0    0    0
   57           0.17           0.00    0    0    2    2    2    0    0    0    0    0    0    0    0    0
   58           0.13           0.00    2    0    2    0    2    0    0    0    0    0    0    0    0    0
   59          -0.13           0.00    2    0    0    0    0    0    0    0    0    0    0    0    0    0
   60          -0.12           0.00    1    0    2   -2    2    0    0    0    0    0    0    0    0    0
   61          -0.11           0.00    0    0    2    0    0    0    0    0    0    0    0    0    0    0];
 
%j = 3  Nb of terms = 4
c3=[...
   62           0.30         -23.51    0    0    0    0    1    0    0    0    0    0    0    0    0    0
   63          -0.03          -1.39    0    0    2   -2    2    0    0    0    0    0    0    0    0    0
   64          -0.01          -0.24    0    0    2    0    2    0    0    0    0    0    0    0    0    0
   65           0.00           0.22    0    0    0    0    2    0    0    0    0    0    0    0    0    0];

%j = 4  Nb of terms = 1
c4=[...  
   66          -0.26          -0.01    0    0    0    0    1    0    0    0    0    0    0    0    0    0];

%-------------------------------------------------------------------------
% FUNDAMENTAL ARGUMENTS
%-------------------------------------------------------------------------
  
    FA = fund_arg (T,2);

%-------------------------------------------------------------------------
%   Evaluate X
%-------------------------------------------------------------------------
    
    Xpol = Xpol * [E;T';T2';T3';T4';T5'];  %[as]

    argx0 = a0(:,4:17)* FA';
    argx1 = a1(:,4:17)* FA';
    argx2 = a2(:,4:17)* FA';
    argx3 = a3(:,4:17)* FA';
    argx4 = a4(:,4:17)* FA';
      
    X =   (a0(:,2))' * sin(argx0) + (a0(:,3))' * cos(argx0)...  
       + ((a1(:,2))' * sin(argx1) + (a1(:,3))' * cos(argx1)).*T'... 
       + ((a2(:,2))' * sin(argx2) + (a2(:,3))' * cos(argx2)).*T2'... 
       + ((a3(:,2))' * sin(argx3) + (a3(:,3))' * cos(argx3)).*T3'... 
       + ((a4(:,2))' * sin(argx4) + (a4(:,3))' * cos(argx4)).*T4';

    X = as2rad(Xpol + X*1e-6);  
    
%-------------------------------------------------------------------------
%   Evaluate Y
%-------------------------------------------------------------------------

    Ypol = Ypol * [E;T';T2';T3';T4';T5'];  %[as]

    argy0 = b0(:,4:17)* FA';
    argy1 = b1(:,4:17)* FA';
    argy2 = b2(:,4:17)* FA';
    argy3 = b3(:,4:17)* FA';
    argy4 = b4(:,4:17)* FA';
       
    Y =    (b0(:,2))' * sin(argy0) + (b0(:,3))' * cos(argy0)     ...  
        + ((b1(:,2))' * sin(argy1) + (b1(:,3))' * cos(argy1)).*T' ... 
        + ((b2(:,2))' * sin(argy2) + (b2(:,3))' * cos(argy2)).*T2'... 
        + ((b3(:,2))' * sin(argy3) + (b3(:,3))' * cos(argy3)).*T3'... 
        + ((b4(:,2))' * sin(argy4) + (b4(:,3))' * cos(argy4)).*T4';
    
    Y = as2rad(Ypol + Y*1e-6);  
    
%-------------------------------------------------------------------------
%   Evaluate S
%-------------------------------------------------------------------------

    Spol = Spol * [E;T';T2';T3';T4';T5'];  %[as]

    args0 = c0(:,4:17)* FA';
    args1 = c1(:,4:17)* FA';
    args2 = c2(:,4:17)* FA';
    args3 = c3(:,4:17)* FA';
    args4 = c4(:,4:17)* FA';
      
    S =    (c0(:,2))' * sin(args0) + (c0(:,3))' * cos(args0)      ...  
        + ((c1(:,2))' * sin(args1) + (c1(:,3))' * cos(args1)).*T' ... 
        + ((c2(:,2))' * sin(args2) + (c2(:,3))' * cos(args2)).*T2'... 
        + ((c3(:,2))' * sin(args3) + (c3(:,3))' * cos(args3)).*T3'... 
        + ((c4(:,2))' * sin(args4) + (c4(:,3))' * cos(args4)).*T4';
    
    S = as2rad((Spol + S)*1e-6); 
    S = S - X.*Y./2;
